import {forwardRef, MouseEventHandler, useCallback} from 'react';

import {TRAIN_COACH_TYPE} from 'projects/trains/constants/coachType';

import IPrice from 'utilities/currency/PriceInterface';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import Price from 'components/Price/Price';
import LabelNonRefundableDiscount from 'projects/trains/components/TrainsOrderPage/LabelNonRefundableDiscount/LabelNonRefundableDiscount';

import cx from './CoachTypeButton.scss';

export type TCoachTypeButtonSize = 'm' | 'l';

export interface ICoachTypeButton {
    value: TRAIN_COACH_TYPE | string;
    title: string;
    price: IPrice;
    disabled?: boolean;
    hasDiscount?: boolean;
    size?: TCoachTypeButtonSize;
}

export interface ICoachTypeButtonProps
    extends ICoachTypeButton,
        IWithQaAttributes {
    active: boolean;
    onClick(): void;
}

const CoachTypeButton = forwardRef<HTMLButtonElement, ICoachTypeButtonProps>(
    function CoachTypeButton(props, ref) {
        const {
            price,
            title,
            active,
            onClick,
            disabled,
            size = 'm',
            hasDiscount,
        } = props;
        const handleClick: MouseEventHandler<HTMLButtonElement> = useCallback(
            event => {
                event.preventDefault();

                onClick();
            },
            [onClick],
        );

        return (
            <button
                ref={ref}
                className={cx('root', `root_size_${size}`, {
                    root_active: active,
                    root_disabled: disabled,
                })}
                onClick={handleClick}
                {...prepareQaAttributes(props)}
            >
                <span
                    className={cx('title')}
                    {...prepareQaAttributes({parent: props, current: 'title'})}
                >
                    {title}
                </span>
                <Price className={cx('price')} {...price} isFrom isRound />
                {hasDiscount && (
                    <LabelNonRefundableDiscount className={cx('discount')} />
                )}
            </button>
        );
    },
);

export default CoachTypeButton;
