import {TrainsPassengersCount} from 'reducers/trains/order/types';
import {ICoachesTypeGroup} from 'projects/trains/components/TrainsOrderPage/SimpleSelectorView/types/TCoachesTypeGroups';

import IPrice from 'utilities/currency/PriceInterface';
import getCoachTypeText from 'projects/trains/lib/getCoachTypeText';
import capitalizeFirstLetter from 'utilities/strings/capitalizeFirstLetter';
import {checkHasCoachesNonRefundableDiscount} from 'projects/trains/lib/order/coaches/checkHasCoachesNonRefundableDiscount';
import {getCoachTabMinPrice} from 'projects/trains/components/TrainsOrderPage/SimpleSelectorView/utilities/getCoachTabMinPrice';
import {checkCoachTabDisabled} from 'projects/trains/components/TrainsOrderPage/SimpleSelectorView/utilities/checkCoachTabDisabled';
import {getCoachTabTitleWithTotalPlaces} from 'projects/trains/components/TrainsOrderPage/SimpleSelectorView/utilities/getCoachTabTitleWithTotalPlaces';
import {calculateCoachTabTotalPlaces} from 'projects/trains/components/TrainsOrderPage/SimpleSelectorView/utilities/calculateCoachTabTotalPlaces';

interface ICalculateCoachTabsOptionsProps {
    isClassSelector?: boolean;
    groups: ICoachesTypeGroup[];
    trainsDisableCoachTypeTabs?: boolean;
    activeGroup?: ICoachesTypeGroup | null;
    passengers: TrainsPassengersCount;
}

export interface ITrainsTabsOption {
    value: string;
    title: string;
    price: IPrice;
    hasDiscount: boolean;
    totalPlaces: number;
    disabled?: boolean;
}

export const calculateCoachTabsOptions = (
    props: ICalculateCoachTabsOptionsProps,
): ITrainsTabsOption[] => {
    const {
        isClassSelector,
        activeGroup,
        groups,
        passengers,
        trainsDisableCoachTypeTabs,
    } = props;

    if (isClassSelector && activeGroup) {
        return activeGroup.classes.map<ITrainsTabsOption>(coachesClass => {
            const {coaches} = coachesClass;
            const totalPlaces = calculateCoachTabTotalPlaces(coaches);

            return {
                value: coachesClass.key,
                price: getCoachTabMinPrice(coachesClass),
                title: getCoachTabTitleWithTotalPlaces({
                    groupTitle: coachesClass.title,
                    totalPlaces,
                }),
                disabled: checkCoachTabDisabled({
                    coaches: coaches,
                    passengers,
                    trainsDisableCoachTypeTabs,
                }),
                hasDiscount: checkHasCoachesNonRefundableDiscount({
                    coaches,
                }),
                totalPlaces,
            };
        });
    } else if (groups.length) {
        return groups.map(coachesTypeGroup => {
            const [firstCoachesClass] = coachesTypeGroup.classes;
            const {coaches} = coachesTypeGroup;
            const totalPlaces = calculateCoachTabTotalPlaces(coaches);

            return {
                value: coachesTypeGroup.type,
                price: getCoachTabMinPrice(firstCoachesClass),
                title: getCoachTabTitleWithTotalPlaces({
                    groupTitle: capitalizeFirstLetter(
                        getCoachTypeText(coachesTypeGroup.type),
                    ),
                    totalPlaces,
                }),
                disabled: checkCoachTabDisabled({
                    coaches,
                    passengers,
                    trainsDisableCoachTypeTabs,
                }),
                hasDiscount: checkHasCoachesNonRefundableDiscount({
                    coaches,
                }),
                totalPlaces,
            };
        });
    }

    return [];
};
