import React, {Component} from 'react';
import {publish} from 'pubsub-js';

import {
    COMPARTMENT,
    SUITE,
} from 'projects/trains/lib/segments/tariffs/constants';
import {
    GENDER_TYPE,
    MALE_AND_FEMALE_GENDER_TYPES,
} from 'projects/trains/constants/genders';

import {ITrainsCoach} from 'reducers/trains/order/types';
import {isNotNull, isNotUndefined} from 'types/utilities';
import {EPubSubEvent} from 'types/EPubSubEvent';

import {IDevice} from 'reducers/common/commonReducerTypes';

import {
    AVAILABLE,
    PLACE_STATES,
    UNAVAILABLE,
} from 'projects/trains/lib/order/placeMap';
import getCoachClassKey from 'projects/trains/lib/order/getCoachClassKey';
import {getGenderTitle, PLACE_SIZE} from 'projects/trains/lib/svgUtils';
import getCoachPlacesPubsubEventName from 'projects/trains/lib/order/coaches/getCoachPlacesPubsubEventName';
import getCoachGender from 'projects/trains/lib/order/gender/getCoachGender';
import {IPlaceMapItem} from 'projects/trains/lib/order/getPlaceMap';

import cx from './SvgSchema.scss';

if (__CLIENT__) {
    require('hammerjs');
    require('classlist-polyfill');
}

const GENDER_TITLE_LINE_HEIGHT = 8;
const GENDER_TITLE_LINE_GAP = 2;

function stopPropagation(e: MouseEvent): void {
    e.stopPropagation();
}

function preventDefault(e: TouchEvent): void {
    e.preventDefault();
}

function getPlaceNumbers(node: HTMLElement): number[] {
    const {id} = node;

    if (!id) {
        return [];
    }

    const numbers = id.split('/').pop()?.split('-').pop()?.split(',');

    if (!numbers) {
        return [];
    }

    return numbers.map(Number);
}

function concatPlaceNumbers(numbers: number[]): string {
    return numbers.sort().join('_');
}

export interface ISchemaPlaceCoords {
    left: number;
    top: number;
    width: number;
    height: number;
}

export interface ISchemaPlaceNodeMapItem {
    node: HTMLElement;
    triggerNode: HTMLElement;
    place: IPlaceMapItem;
    numbers: number[];
    coords: ISchemaPlaceCoords | null;
}

interface IGenderHint {
    gender: GENDER_TYPE | null;
    genderTipUp: null | boolean;
    left: number;
    top: number;
    right: number;
    bottom: number;
}

interface ISvgSchemaProps {
    schema: string;

    /**
     * Уникальный префикс, который добавляется к svg mask'ам мест,
     * чтобы было корректное отображение нескольких одинаковых схем.
     */
    schemaPlaceMaskPrefix: string;
    coach: ITrainsCoach;
    gender: GENDER_TYPE | null;
    placeMap: IPlaceMapItem[];
    interactive: boolean;
    deviceType: IDevice;
    coachSchemaRenderingDelay?: number;
    onPlaceCoordsCalculated(
        placeNodeMap: PartialRecord<string, ISchemaPlaceNodeMapItem>,
    ): void;
    onPlaceClick({
        places,
        coords,
    }: {
        places: IPlaceMapItem[];
        coords: ISchemaPlaceCoords;
    }): void;
}

interface ISvgSchemaState {
    shouldShowAnimatedSkeleton: boolean;
}

export default class SvgSchema extends Component<
    ISvgSchemaProps,
    ISvgSchemaState
> {
    state = {
        shouldShowAnimatedSkeleton: Boolean(
            this.props.coachSchemaRenderingDelay,
        ),
    };

    private hideAnimatedSkeletonTimeout: number | undefined = undefined;
    private setCoordsTimeout: number | undefined = undefined;
    private createGenderHintsTimeout: number | undefined = undefined;
    private _placesContainer: HTMLElement | null = null;
    private _textContainer: HTMLElement | null = null;
    private _placeNodeMap: PartialRecord<
        string,
        ISchemaPlaceNodeMapItem
    > | null = null;
    private _selfRef = React.createRef<HTMLDivElement>();
    private hammer: HammerManager | null = null;

    componentDidMount(): void {
        if (this.state.shouldShowAnimatedSkeleton) {
            this.hideAnimatedSkeletonTimeout = window.setTimeout(
                () => this.setState({shouldShowAnimatedSkeleton: false}),
                this.props.coachSchemaRenderingDelay,
            );

            return;
        }

        this.init();
    }

    // eslint-disable-next-line camelcase
    UNSAFE_componentWillReceiveProps(nextProps: ISvgSchemaProps): void {
        const {
            schema: nextPropsSchema,
            gender: nextPropsGender,
            placeMap: nextPropsPlaceMap,
        } = nextProps;

        if (
            this.props.schema === nextPropsSchema &&
            !this.state.shouldShowAnimatedSkeleton
        ) {
            this.updateState(nextPropsPlaceMap, nextPropsGender);
        }
    }

    shouldComponentUpdate(
        nextProps: ISvgSchemaProps,
        nextState: ISvgSchemaState,
    ): boolean {
        const {schema: nextPropsSchema} = nextProps;

        return (
            this.props.schema !== nextPropsSchema ||
            this.state.shouldShowAnimatedSkeleton !==
                nextState.shouldShowAnimatedSkeleton ||
            getCoachClassKey({coach: this.props.coach}) !==
                getCoachClassKey({coach: nextProps.coach}) ||
            this.props.coach.arrival !== nextProps.coach.arrival
        );
    }

    componentDidUpdate(): void {
        if (this.state.shouldShowAnimatedSkeleton) {
            return;
        }

        this.init();
    }

    componentWillUnmount(): void {
        window.clearTimeout(this.hideAnimatedSkeletonTimeout);
        window.clearTimeout(this.setCoordsTimeout);
        window.clearTimeout(this.createGenderHintsTimeout);

        this.unsubscribeEvents();
    }

    private subscribeEvents(): void {
        const {deviceType} = this.props;

        if (!this._placesContainer) {
            return;
        }

        if (deviceType.isDesktop) {
            this._placesContainer.addEventListener(
                'click',
                this.onPlaceClick as EventListener,
            );
        } else {
            this.hammer = new Hammer(this._placesContainer, {
                touchAction: 'auto',
            });
            // @ts-ignore
            this.hammer.on('tap', this.onPlaceClick);
            this._placesContainer.addEventListener('touchend', preventDefault);
            document.addEventListener('click', this.onMouseLeave);
        }

        this._placesContainer.addEventListener(
            'mousemove',
            this.onMouseMove as EventListener,
        );
        this._placesContainer.addEventListener(
            'mouseover',
            this.onMouseOver as EventListener,
        );
        this._placesContainer.addEventListener('mouseleave', this.onMouseLeave);
    }

    private unsubscribeEvents(): void {
        const {isMobile} = this.props.deviceType;

        if (this._placesContainer) {
            this._placesContainer.removeEventListener(
                'click',
                this.onPlaceClick as EventListener,
            );
            this._placesContainer.removeEventListener('click', stopPropagation);
            this._placesContainer.removeEventListener(
                'mousemove',
                this.onMouseMove as EventListener,
            );
            this._placesContainer.removeEventListener(
                'mouseover',
                this.onMouseOver as EventListener,
            );
            this._placesContainer.removeEventListener(
                'mouseleave',
                this.onMouseLeave,
            );

            if (isMobile) {
                this._placesContainer.removeEventListener(
                    'touchend',
                    preventDefault,
                );
            }
        }

        if (this.hammer) {
            this.hammer.destroy();
        }

        if (isMobile) {
            document.removeEventListener('click', this.onMouseLeave);
        }
    }

    /*
     * Метод добавляет всю необходимую для работы со схемой информацию при инициализации
     */
    private init(): void {
        const {gender, schemaPlaceMaskPrefix, placeMap} = this.props;

        if (!this._selfRef.current) {
            return;
        }

        this._textContainer = this._selfRef.current.querySelector('#base');
        this._placesContainer = this._selfRef.current.querySelector('#places');

        const places: HTMLElement[] = Array.from(
            this._selfRef.current.querySelectorAll('#places > g'),
        );

        this._placeNodeMap = places.reduce<
            PartialRecord<string, ISchemaPlaceNodeMapItem>
        >((nodeMap, node) => {
            const trigger = node.querySelector<HTMLElement>('[id^="trigger"]');
            const numbers = getPlaceNumbers(node);
            const place = placeMap.find(item => numbers.includes(item.number));
            const mask = node.querySelector('mask');
            const pathAfterMask = node.querySelector('mask + path');

            if (!trigger || !place) {
                return nodeMap;
            }

            if (mask && pathAfterMask) {
                const newMaskId = `${schemaPlaceMaskPrefix}${concatPlaceNumbers(
                    numbers,
                )}`;

                // нужно выставлять уникальные идентификаторы, т.к. они повторяются при нескольких схемах на странице
                mask.setAttribute('id', newMaskId);
                pathAfterMask.setAttribute('mask', `url(#${newMaskId})`);
            }

            // убираем закраску у элементов, т.к. у нас есть своя закраска, которая должна унаследоваться от элементов-предков
            const rectAndPathElements = node.querySelectorAll('rect, path');

            // @ts-ignore
            for (const el of rectAndPathElements) {
                el.removeAttribute('fill');
            }

            // При обновлении компонента надо подчищать неактуальные имена классов
            // т.к. классами тут рулим мы а не React
            if (node.className) {
                node.setAttribute('class', '');
            }

            node.classList.add('Place');
            node.classList.add(`Place_${concatPlaceNumbers(numbers)}`);

            trigger.classList.add('Place__trigger');

            if (place.theme) {
                node.classList.add(`Place_theme_${place.theme}`);
            }

            if (place.gender && place.gender !== GENDER_TYPE.SINGLE) {
                node.classList.add(`Place_${place.gender}`);
            }

            nodeMap[concatPlaceNumbers(numbers)] = {
                node,
                triggerNode: trigger,
                place,
                numbers,
                coords: null,
            };

            return nodeMap;
        }, {});

        this.setPlaceNodesCoords();

        this.createGenderHintsTimeout = window.setTimeout(
            this.createGenderHints,
            0,
        );

        this.updateState(placeMap, gender);

        this.unsubscribeEvents();
        this.subscribeEvents();
    }

    /*
     * Метод выставляет координаты нод мест
     */
    private setPlaceNodesCoords = (): void => {
        if (!this._selfRef.current || !this._placeNodeMap) {
            return;
        }

        const {onPlaceCoordsCalculated} = this.props;

        const containerCoords = this._selfRef.current.getBoundingClientRect();

        Object.values(this._placeNodeMap).forEach(nodeInfo => {
            if (!nodeInfo) {
                return;
            }

            const position = nodeInfo.triggerNode.getBoundingClientRect();

            nodeInfo.coords = {
                left: position.left - containerCoords.left,
                top: position.top - containerCoords.top,
                height: position.height,
                width: position.width,
            };
        });

        if (onPlaceCoordsCalculated) {
            onPlaceCoordsCalculated(this._placeNodeMap);
        }
    };

    /*
     * Метод получает и отображает гендерные подсказки, если они необходимы
     */
    private createGenderHints = (): void => {
        const {coach} = this.props;

        const textContainer = this._textContainer;

        if (!textContainer) {
            return;
        }

        const genderHints = Array.from(
            textContainer.querySelectorAll('.Place__genderTitle'),
        );

        // Т.к. мы работаем с одной и той же dom нодой, то нужно подчищать за собой артефакты
        genderHints.forEach(node => textContainer.removeChild(node));

        if (
            (coach.type === SUITE || coach.type === COMPARTMENT) &&
            getCoachGender(coach)
        ) {
            this.renderGenderHints(this.getGenderHints());
        }
    };

    /*
     * Метод получает информацию для отображения гендерных подсказок
     */
    private getGenderHints(): IGenderHint[] {
        const {placeMap} = this.props;

        const placeNodeMap = this._placeNodeMap;

        if (!placeNodeMap) {
            return [];
        }

        const places = Object.values(placeMap).filter(
            ({placeState}) =>
                placeState === AVAILABLE || placeState === UNAVAILABLE,
        );
        const groups = [...new Set(places.map(({groupNumber}) => groupNumber))];

        return groups.map(number => {
            const groupPlaces = Object.values(placeNodeMap)
                .filter(placeNode => placeNode?.place.groupNumber === number)
                .filter(isNotUndefined);
            const placesCoords = groupPlaces
                .map(({coords}) => coords)
                .filter(isNotNull);

            return {
                gender: groupPlaces.reduce<GENDER_TYPE | null>(
                    (gender, {place}) => gender || place.gender,
                    null,
                ),
                genderTipUp: groupPlaces.reduce<null | boolean>(
                    (genderTipUp, {place}) => genderTipUp || place.genderTipUp,
                    null,
                ),
                left: Math.min(...placesCoords.map(({left}) => left)),
                top: Math.min(...placesCoords.map(({top}) => top)),
                right: Math.max(
                    ...placesCoords.map(({left}) => left + PLACE_SIZE),
                ),
                bottom: Math.max(
                    ...placesCoords.map(({top}) => top + PLACE_SIZE),
                ),
            };
        });
    }

    /*
     * Метод добавляет гендерные подскаки к схеме
     */
    private renderGenderHints(hints: IGenderHint[]): void {
        const textContainer = this._textContainer;

        if (!textContainer) {
            return;
        }

        hints.forEach(
            ({gender, genderTipUp, left, top, right, bottom}) =>
                gender &&
                gender !== GENDER_TYPE.SINGLE &&
                textContainer.appendChild(
                    getGenderTitle(gender, {
                        x: left + (right - left) / 2,
                        y: genderTipUp
                            ? top -
                              2 *
                                  (GENDER_TITLE_LINE_HEIGHT +
                                      GENDER_TITLE_LINE_GAP)
                            : bottom + 2 * GENDER_TITLE_LINE_HEIGHT,
                    }),
                ),
        );
    }

    /*
     * Метод обновляет классы состояния у нод мест
     */
    private updateState(
        places: IPlaceMapItem[],
        coachGender: GENDER_TYPE | null,
    ): void {
        const placeNodeMap = this._placeNodeMap;

        if (!placeNodeMap) {
            return;
        }

        Object.keys(placeNodeMap).forEach(placeNodeKey => {
            const numbers = placeNodeKey.split('_').map(Number);

            numbers.forEach(number => {
                const currentPlace = places.find(
                    place => place.number === number,
                );

                const placeNodeItem = placeNodeMap[placeNodeKey];

                if (!currentPlace || !placeNodeItem) {
                    return;
                }

                const {gender, placeState} = currentPlace;
                const {classList} = placeNodeItem.node;

                if (gender === GENDER_TYPE.SINGLE) {
                    const dynamicGender = coachGender || GENDER_TYPE.SINGLE;

                    for (
                        let j = 0;
                        j < MALE_AND_FEMALE_GENDER_TYPES.length;
                        j++
                    ) {
                        classList.toggle(
                            `Place_${MALE_AND_FEMALE_GENDER_TYPES[j]}`,
                            dynamicGender === MALE_AND_FEMALE_GENDER_TYPES[j],
                        );
                    }
                }

                for (let j = 0; j < PLACE_STATES.length; j++) {
                    classList.toggle(
                        `Place_${PLACE_STATES[j]}`,
                        placeState === PLACE_STATES[j],
                    );
                }
            });
        });
    }

    private getPlaceNode(e: MouseEvent & {target: Node}): Node {
        let target = e.target;

        while (
            target.parentNode &&
            target.parentNode !== this._placesContainer
        ) {
            target = target.parentNode;
        }

        return target;
    }

    private onPlaceClick = (e: MouseEvent & {target: Node}): void => {
        const {placeMap, onPlaceClick, deviceType} = this.props;
        const placeNode = this.getPlaceNode(e);

        const placeNodeMap = this._placeNodeMap;

        if (!placeNode || !placeNodeMap) {
            return;
        }

        const numbers = getPlaceNumbers(placeNode as HTMLElement);
        const places = placeMap.filter(item => numbers.includes(item.number));

        const coords = placeNodeMap[concatPlaceNumbers(numbers)]?.coords;

        if (places.length > 0 && coords) {
            onPlaceClick({
                places,
                coords,
            });

            e.stopPropagation?.();

            if (deviceType.isMobile) {
                this.publishTooltip(EPubSubEvent.SCHEME_CLOSE_ALL);

                this.publishTooltip(EPubSubEvent.SCHEME_PLACE_ENTER, places);
            }
        }
    };

    private getPlacesFromEvent(
        e: MouseEvent & {target: Node},
    ): IPlaceMapItem[] {
        const placeNode = this.getPlaceNode(e);

        if (placeNode) {
            const numbers = getPlaceNumbers(placeNode as HTMLElement);
            const {placeMap} = this.props;

            return placeMap.filter(item => numbers.includes(item.number));
        }

        return [];
    }

    private handleMouseAction(
        e: MouseEvent & {target: Node},
        pubSubEvent: EPubSubEvent,
    ): void {
        const places = this.getPlacesFromEvent(e);

        this.publishTooltip(pubSubEvent, places);
    }

    private publishTooltip(
        pubSubEvent: EPubSubEvent,
        places?: IPlaceMapItem[],
    ): void {
        if (pubSubEvent === EPubSubEvent.SCHEME_CLOSE_ALL) {
            publish(pubSubEvent, null);

            return;
        }

        const placeNodeMap = this._placeNodeMap;

        let payload: {
            place: IPlaceMapItem;
            coords: ISchemaPlaceCoords;
        } | null = null;

        if (places && places.length > 0 && placeNodeMap) {
            const coords =
                placeNodeMap[
                    concatPlaceNumbers(places.map(place => place.number))
                ]?.coords;

            if (coords) {
                // Если выбираем всё купе целиком, то принимаем за инстину, что все места имеют одинаковое состояние
                payload = {place: places[0], coords};
            }
        }

        publish(
            getCoachPlacesPubsubEventName(pubSubEvent, this.props.coach),
            payload,
        );
    }

    private onMouseMove = (e: MouseEvent & {target: Node}): void => {
        this.handleMouseAction(e, EPubSubEvent.SCHEME_MOUSE_MOVE);
    };

    private onMouseOver = (e: MouseEvent & {target: Node}): void => {
        this.handleMouseAction(e, EPubSubEvent.SCHEME_PLACE_ENTER);
    };

    private onMouseLeave = (): void => {
        this.publishTooltip(EPubSubEvent.SCHEME_PLACE_LEAVE);
    };

    render(): React.ReactNode {
        const {shouldShowAnimatedSkeleton} = this.state;
        const {schema, interactive} = this.props;

        if (shouldShowAnimatedSkeleton) {
            const schemaWidth = Math.min(
                Number(/width="(\d+)(px)?"/.exec(schema)?.[1] || 0),
                760,
            );
            const schemaHeight = Number(
                /height="(\d+)(px)?"/.exec(schema)?.[1] || 0,
            );

            return (
                <div
                    className={cx('root', {root_animated: true})}
                    style={{
                        width: `${schemaWidth}px`,
                        height: `${schemaHeight}px`,
                    }}
                    ref={this._selfRef}
                />
            );
        }

        return (
            <div
                className={cx('root', {root_static: !interactive})}
                ref={this._selfRef}
                dangerouslySetInnerHTML={{__html: schema}}
            />
        );
    }
}
