import React, {Component, ReactNode} from 'react';
import {Route, RouteComponentProps, Switch, withRouter} from 'react-router-dom';
import {connect} from 'react-redux';
import arrayMutators from 'final-form-arrays';
import setFieldData from 'final-form-set-field-data';

import {EProjectName} from 'constants/common';
import {URLs} from 'constants/urls';
import {EFormKey} from 'constants/form/EFormKey';
import {ORDER_STEP} from 'projects/trains/constants/orderSteps';
import {ORDER_PAGE_STATUSES} from 'projects/trains/constants/orderPageStatuses';
import {validationInfo} from 'projects/trains/constants/booking/validation/validationInfo';

import {ECommerceActionType} from 'utilities/metrika/types/ecommerce';
import {isFilledTrainsSearchContext} from 'reducers/trains/context/types';
import {EFooterProject} from 'components/Footer/types';
import {ITrainsBookFormValues} from 'types/trains/booking/ITrainsBookFormValues';

import createOrderAction from 'reducers/trains/order/thunk/createOrder/createOrder';
import changeOrderStepAction from 'reducers/trains/order/thunk/changeOrderStep';
import setOrderContactsAction from 'reducers/trains/order/thunk/setOrderContacts';
import setOrderPassengersAction from 'reducers/trains/order/thunk/setOrderPassengers';
import {setOrderUseContacts} from 'reducers/trains/order/actions/contacts';
import sendMetrikaAfterCreateOrderAction from 'reducers/trains/order/thunk/sendMetrikaAfterCreateOrder';

import orderDataSelector from 'selectors/trains/orderDataSelector';

import {deviceMods} from 'utilities/stylesUtils';
import {eCommercePush} from 'utilities/metrika/ecommerce';
import renderOrderStep from 'projects/trains/components/TrainsOrderApp/helpers/renderOrderStep';
import {
    getOrderError,
    IPartnerError,
} from 'projects/trains/lib/errors/orderErrors';
import getECommerceProductFromContext from 'projects/trains/lib/metrika/getECommerceProductFromContext';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {reachValidationGoal} from './utilities/reachValidationGoal';

import * as i18nBlock from 'i18n/trains-details-loading';

import Box from 'components/Box/Box';
import Form from 'components/Form/Form';
import focusFirstInvalidField from 'components/Form/mutators/focusFirstInvalidField';
import createSubmitErrorDecorator from 'components/Form/decorators/createSubmitErrorDecorator';
import Container from 'components/Layouts/Container/Container';
import LayoutDefault from 'components/Layouts/LayoutDefault/LayoutDefault';
import BookingLayout from 'components/Layouts/BookingLayout/BookingLayout';
import NotFoundFragment from 'components/NotFound/NotFoundFragment/NotFoundFragment';
import {TrainsOrderMeta} from 'projects/trains/components/meta/TrainsOrderMeta/TrainsOrderMeta';
import TrainsOrderSteps from 'projects/trains/components/OrderSteps/OrderSteps';
import OrderSummary from 'projects/trains/components/OrderSummary/OrderSummary';
import TrainsOrderSegments from 'projects/trains/components/TrainsOrderSegments/TrainsOrderSegments';
import TrainsOrderErrorPage from 'projects/trains/components/TrainsOrderPage/ErrorPage/ErrorPage';
import LegalInformation from 'projects/trains/components/TrainsOrderPage/LegalInformation/LegalInformation';
import ConfirmationTimer from 'projects/trains/components/TrainsOrderPage/OrderTimer/ConfirmationTimer';
import ConfirmationTimerSkeleton from 'projects/trains/components/TrainsOrderPage/OrderTimer/ConfirmationTimerSkeleton';
import PlacesStepSkeleton from 'projects/trains/components/TrainsOrderPage/components/PlacesStepSkeleton/PlacesStepSkeleton';
import PassengersStepSkeleton from 'projects/trains/components/TrainsOrderPage/components/PassengersStepSkeleton/PassengersStepSkeleton';
import ConfirmStepSkeleton from 'projects/trains/components/TrainsOrderPage/components/ConfirmStepSkeleton/ConfirmStepSkeleton';
import BookLoader from 'components/BookLoader/BookLoader';

import PartnersRequisites from './PartnersRequisites/PartnersRequisites';
import CancelOrderButton from './CancelOrderButton/CancelOrderButton';
import {CONFIRM_STEP_QA} from './ConfirmStep/ConfirmStepContainer';

import cx from './TrainsOrderPage.scss';

const mapDispatchToProps = {
    createOrder: createOrderAction,
    changeOrderStep: changeOrderStepAction,
    setOrderContacts: setOrderContactsAction,
    setOrderPassengers: setOrderPassengersAction,
    setOrderUseMainContacts: setOrderUseContacts,
    sendMetrikaAfterCreateOrder: sendMetrikaAfterCreateOrderAction,
};

const focusOnSubmitErrors = createSubmitErrorDecorator<ITrainsBookFormValues>(
    EFormKey.TRAIN_BOOK,
    reachValidationGoal,
);

type TTrainsOrderPageProps = ReturnType<typeof orderDataSelector> &
    typeof mapDispatchToProps &
    RouteComponentProps;

const ROOT_QA = 'trainsOrderPage';

class TrainsOrderPage extends Component<TTrainsOrderPageProps> {
    componentDidMount(): void {
        const {context} = this.props;

        if (isFilledTrainsSearchContext(context)) {
            eCommercePush(ECommerceActionType.ADD, [
                getECommerceProductFromContext(context),
            ]);
        }
    }

    private async createOrder(): Promise<void> {
        const {
            createOrder,
            changeOrderStep,
            nextOrderStep,
            sendMetrikaAfterCreateOrder,
        } = this.props;

        if (await createOrder()) {
            changeOrderStep(nextOrderStep);

            sendMetrikaAfterCreateOrder();
        }
    }

    /* Handlers */

    /**
     * TODO: Переместить на всех табах обработку из TrainsOrderButton сюда, там оставить только метрики OnClick.
     */
    private handleFormSubmit = (formValues: ITrainsBookFormValues): void => {
        const {setOrderContacts, setOrderPassengers, setOrderUseMainContacts} =
            this.props;

        setOrderPassengers(formValues);
        setOrderContacts(formValues.contacts);
        setOrderUseMainContacts(
            Boolean(formValues.contacts?.useOnlyMainContacts),
        );

        this.createOrder();
    };

    /* Render */

    private renderSkeleton(): React.ReactNode {
        const {deviceType, orderStep} = this.props;

        switch (orderStep) {
            case ORDER_STEP.PLACES:
                return <PlacesStepSkeleton deviceType={deviceType} />;
            case ORDER_STEP.CONFIRM:
                return <ConfirmStepSkeleton deviceType={deviceType} />;
            case ORDER_STEP.PASSENGERS:
                return <PassengersStepSkeleton deviceType={deviceType} />;
            default:
                return null;
        }
    }

    private renderLeftColumnContent(
        errors: boolean,
        partnerError: IPartnerError | null,
    ): ReactNode {
        const {orderPageStatus, orderStep} = this.props;

        if (orderPageStatus === ORDER_PAGE_STATUSES.FETCHING_DATA) {
            return this.renderSkeleton();
        }

        if (errors) {
            return (
                <TrainsOrderErrorPage
                    errorData={getOrderError(partnerError)}
                    withoutAction
                />
            );
        }

        return (
            <Switch>
                <Route
                    path={URLs.trainsOrderPassengers}
                    render={renderOrderStep(orderStep, ORDER_STEP.PASSENGERS)}
                />

                <Route
                    path={URLs.trainsOrderConfirm}
                    render={renderOrderStep(orderStep, ORDER_STEP.CONFIRM)}
                />

                {/* порядок важен, т.к. у страницы выбора мест наименьшая специфичность */}
                <Route
                    path={URLs.trainsOrder}
                    render={renderOrderStep(orderStep, ORDER_STEP.PLACES)}
                />

                <Route component={NotFoundFragment} />
            </Switch>
        );
    }

    private renderLeftColumn(
        errors: boolean,
        partnerError: IPartnerError | null,
    ): ReactNode {
        const {
            orderPageStatus,
            orderStep,
            trainSegments,
            currentSegmentDirectionAndIndex,
            orderTripInfo,
        } = this.props;

        if (errors && orderStep === ORDER_STEP.CONFIRM) {
            return null;
        }

        return (
            <>
                {orderStep !== ORDER_STEP.CONFIRM && (
                    <BookingLayout.Snippets>
                        <TrainsOrderSegments
                            segments={trainSegments}
                            tripInfo={orderTripInfo}
                            loading={
                                orderPageStatus ===
                                ORDER_PAGE_STATUSES.FETCHING_DATA
                            }
                            activeSegmentDirectionAndIndex={
                                orderStep === ORDER_STEP.PLACES
                                    ? currentSegmentDirectionAndIndex
                                    : undefined
                            }
                            canHideCompanyForMobileView
                            withChangeTrainLink
                            {...prepareQaAttributes('trainsOrderSegments')}
                        />
                    </BookingLayout.Snippets>
                )}
                {this.renderLeftColumnContent(Boolean(errors), partnerError)}
                {orderPageStatus === ORDER_PAGE_STATUSES.RESERVING_TICKETS && (
                    <BookLoader
                        className={cx('loader')}
                        isModalView
                        isLoading
                        title={i18nBlock.purchaseDashPageDashReservingDashTickets()}
                    />
                )}
            </>
        );
    }

    private renderDisclaimers(): ReactNode {
        const {orderStep} = this.props;

        return (
            <BookingLayout.CartCaption>
                {orderStep === ORDER_STEP.CONFIRM && (
                    <CancelOrderButton
                        className={cx('cancelOrder')}
                        {...prepareQaAttributes(CONFIRM_STEP_QA)}
                    />
                )}

                <LegalInformation />

                {orderStep === ORDER_STEP.CONFIRM && (
                    <PartnersRequisites
                        className={cx('partnersRequisites')}
                        {...prepareQaAttributes({
                            parent: CONFIRM_STEP_QA,
                            current: 'partnersRequisites',
                        })}
                    />
                )}
            </BookingLayout.CartCaption>
        );
    }

    private renderRightColumn(): ReactNode {
        const {
            deviceType,
            currentOrderPlaces,
            orderPageStatus,
            orderStep,
            orderInfo,
        } = this.props;
        const {isMobile, isDesktop} = deviceType;
        const isReady = orderPageStatus !== ORDER_PAGE_STATUSES.FETCHING_DATA;

        /**
         * В мобильной версии не показываем корзинку:
         * когда выбрано место в вагоне или пока идет загрузка
         */
        if (
            isMobile &&
            ((orderStep === ORDER_STEP.PLACES &&
                currentOrderPlaces.length !== 0) ||
                !isReady)
        ) {
            return null;
        }

        return (
            <>
                <Box>
                    {isDesktop &&
                        orderStep === ORDER_STEP.CONFIRM &&
                        (orderInfo?.expiresAt ? (
                            <ConfirmationTimer
                                className={cx('timer')}
                                deviceType={deviceType}
                                orderInfo={orderInfo}
                            />
                        ) : (
                            <ConfirmationTimerSkeleton
                                className={cx('timer')}
                                deviceType={deviceType}
                            />
                        ))}

                    <BookingLayout.Cart className={cx('summary')}>
                        <OrderSummary
                            hasGoToConfirmStep={
                                orderStep === ORDER_STEP.CONFIRM
                            }
                            {...prepareQaAttributes('pageOrderSummary')}
                        />
                    </BookingLayout.Cart>
                </Box>

                {isReady && this.renderDisclaimers()}
            </>
        );
    }

    private renderContent(): ReactNode {
        const {deviceType, trainDetailsError, orderPageStatus} = this.props;

        const {errors, partnerError} = trainDetailsError;

        return (
            <Form<ITrainsBookFormValues>
                validationInfo={validationInfo}
                subscription={{}}
                decorators={[focusOnSubmitErrors]}
                mutators={{
                    ...arrayMutators,
                    setFieldData,
                    focusFirstInvalidField,
                }}
                onSubmit={this.handleFormSubmit}
                render={({handleSubmit}): ReactNode => (
                    <form
                        name={EFormKey.TRAIN_BOOK}
                        onSubmit={handleSubmit}
                        autoComplete="off"
                        noValidate
                    >
                        <TrainsOrderSteps
                            {...prepareQaAttributes('orderSteps')}
                        />
                        <BookingLayout
                            className={cx('layout', {
                                layout_loading:
                                    orderPageStatus ===
                                    ORDER_PAGE_STATUSES.FETCHING_DATA,
                            })}
                            deviceType={deviceType}
                            leftColumn={this.renderLeftColumn(
                                errors,
                                partnerError,
                            )}
                            rightColumn={!errors && this.renderRightColumn()}
                        />
                    </form>
                )}
            />
        );
    }

    renderError(): ReactNode {
        const {
            trainDetailsError: {partnerError},
        } = this.props;

        return (
            <Container>
                <TrainsOrderErrorPage errorData={getOrderError(partnerError)} />
            </Container>
        );
    }

    render(): ReactNode {
        const {deviceType, trainDetailsError, orderPageStatus, orderStep} =
            this.props;

        const {errors, partnerError} = trainDetailsError;

        if (orderStep === ORDER_STEP.PAYMENT) {
            return null;
        }

        const leftColumnContent = this.renderLeftColumn(errors, partnerError);
        const showError = errors && !leftColumnContent;

        return (
            <>
                <TrainsOrderMeta />

                <LayoutDefault
                    className={cx('root', deviceMods('root', deviceType))}
                    showSearchForm={deviceType.isDesktop}
                    showNavigation={deviceType.isDesktop}
                    showFooter={
                        orderPageStatus !== ORDER_PAGE_STATUSES.FETCHING_DATA
                    }
                    searchFormInitialIsExpanded={false}
                    project={EProjectName.TRAINS}
                    footerType={EFooterProject.TRAINS}
                    {...prepareQaAttributes(ROOT_QA)}
                >
                    {showError ? this.renderError() : this.renderContent()}
                </LayoutDefault>

                <div
                    {...prepareQaAttributes(`${orderStep}-${orderPageStatus}`)}
                />
            </>
        );
    }
}

export default withRouter(
    connect(orderDataSelector, mapDispatchToProps)(TrainsOrderPage),
);
