import React, {Component} from 'react';
import {connect} from 'react-redux';
import {batchActions} from 'redux-batched-actions';
import minBy from 'lodash/minBy';

import {ORDER_STEP} from 'projects/trains/constants/orderSteps';
import {GENDER_TYPE} from 'projects/trains/constants/genders';
import {PLACE_RESERVATION_TYPE} from 'projects/trains/constants/placeReservationType';

import {ITrainsCoach, TrainsPassengersCount} from 'reducers/trains/order/types';
import {ITrainsSchema} from 'server/api/TrainsApi/types/ITrainsDetailsApiResponse';
import {isNotUndefined} from 'types/utilities';
import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {ETrainsGoal} from 'utilities/metrika/types/goals/trains';

import {
    addPlacesToOrder,
    clearSchemeFromCoaches,
    removePlacesFromOrder,
    setPreferredGender,
    setSchemeHasGroupedPlaces,
} from 'reducers/trains/order/actions/trains';
import {StoreInterface} from 'reducers/storeTypes';

import currentSegmentDirectionAndIndexSelector from 'selectors/trains/order/currentSegmentDirectionAndIndexSelector';

import {reachGoalOnce} from 'utilities/metrika';
import {countPassengersWithPlaces} from 'projects/trains/lib/order/passengers/utils';
import {AVAILABLE, SELECTED} from 'projects/trains/lib/order/placeMap';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {IPlaceMapItem} from 'projects/trains/lib/order/getPlaceMap';
import {logError} from 'utilities/logger/logError';

import {TBoxSizes} from 'components/Box/Box';
import GenderPopup from 'projects/trains/components/TrainsOrderPage/GenderPopup/GenderPopup';
import SvgSchemaContainer from 'projects/trains/components/TrainsOrderPage/SvgSchemaContainer/SvgSchemaContainer';
import SvgSchemaLegend from 'projects/trains/components/TrainsOrderPage/SvgSchemaLegend/SvgSchemaLegend';
import {
    ISchemaPlaceCoords,
    ISchemaPlaceNodeMapItem,
} from 'projects/trains/components/TrainsOrderPage/SvgSchema/SvgSchema';
import HorizontalScroller from 'components/HorizontalScroller/HorizontalScroller';

import cx from './TransportSchema.scss';

const SCHEMA_TOP_PADDING = 20;
const SCHEMA_LEFT_PADDING = 36; // Расстояние от svg схемы до края экрана

interface ITransportSchemaStateProps {
    currentSegmentDirectionAndIndex: ReturnType<
        typeof currentSegmentDirectionAndIndexSelector
    >;
}

const mapStateToProps = (
    state: StoreInterface,
): ITransportSchemaStateProps => ({
    currentSegmentDirectionAndIndex:
        currentSegmentDirectionAndIndexSelector(state),
});

const mapDispatchToProps = {
    dispatchActions: batchActions,
    addPlacesToOrder,
    setSchemeHasGroupedPlaces,
    clearSchemeFromCoaches,
};

interface ITransportSchemaOwnProps
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    legendClassName?: string;
    coach: ITrainsCoach;
    passengers: TrainsPassengersCount;
    orderPlaces: number[];
    gender: GENDER_TYPE | null;
    placeMap: IPlaceMapItem[];
    schema: ITrainsSchema;
    coachSchemaRenderingDelay?: number;
    schemeHasGroupedPlaces?: boolean;
    schemaPlaceMaskPrefix: string;
    orderStep?: ORDER_STEP;
    interactive?: boolean;
    scrollOffset?: TBoxSizes;
    onCoachChange?(): void;
    onSchemaCatchError?(): void;
}

interface ITransportSchemaState {
    places: IPlaceMapItem[] | null;
    coords: ISchemaPlaceCoords;
}

type TTransportSchemaProps = ITransportSchemaOwnProps &
    ITransportSchemaStateProps &
    typeof mapDispatchToProps;

class TransportSchema extends Component<
    TTransportSchemaProps,
    ITransportSchemaState
> {
    static defaultProps = {
        interactive: true,
    };

    state: ITransportSchemaState = {
        places: null,
        coords: {
            width: 0,
            height: 0,
            left: 0,
            top: 0,
        },
    };

    private schemaPlaceNodeMap: PartialRecord<
        string,
        ISchemaPlaceNodeMapItem
    > | null = null;
    private rendered: boolean = false;
    private _pseudoPlaceRef = React.createRef<HTMLDivElement>();
    private scrollableContainer = React.createRef<HTMLDivElement>();

    componentDidMount(): void {
        // Если SvgSchema была отрендерена сразу (без отображения анимированного скелетона)
        // то сразу же скролим схему к первому свободному месту на мобилках, чтобы не было морганий при отображении и скроле
        if (this.schemaPlaceNodeMap) {
            this.scrollSchemaToFirstSelectablePlace();
        }

        this.rendered = true;
    }

    componentDidUpdate(prevProps: TTransportSchemaProps): void {
        if (
            this.props.coach !== prevProps.coach ||
            countPassengersWithPlaces(this.props.passengers) !==
                countPassengersWithPlaces(prevProps.passengers)
        ) {
            this.resetPlace();
        }
    }

    onCloseGenderSelector = (): void => {
        this.resetPlace();
    };

    onPlaceClick = ({
        places,
        coords,
    }: {
        places: IPlaceMapItem[];
        coords: ISchemaPlaceCoords;
    }): void => {
        const {
            passengers,
            orderPlaces,
            coach,
            schemeHasGroupedPlaces,
            onCoachChange,
            currentSegmentDirectionAndIndex,
        } = this.props;
        const passengersWithPlaces = countPassengersWithPlaces(passengers);
        const placeNumbers = places.map(place => place.number);
        const isToggleMode =
            passengersWithPlaces === 1 ||
            (coach.placeReservationType !== PLACE_RESERVATION_TYPE.USUAL &&
                schemeHasGroupedPlaces);
        const actions = [];

        onCoachChange?.();

        if (places.every(place => place.placeState === SELECTED)) {
            if (orderPlaces.length === places.length) {
                actions.push(
                    setPreferredGender({
                        ...currentSegmentDirectionAndIndex,
                        data: null,
                    }),
                );

                this.resetPlace();
            }

            actions.push(
                removePlacesFromOrder({
                    ...currentSegmentDirectionAndIndex,
                    data: placeNumbers,
                }),
            );
        }

        if (places.every(place => place.placeState === AVAILABLE)) {
            reachGoalOnce(ETrainsGoal.PLACE_SELECTED);

            if (isToggleMode) {
                actions.push(
                    removePlacesFromOrder({
                        ...currentSegmentDirectionAndIndex,
                        data: orderPlaces,
                    }),
                );
                actions.push(
                    setPreferredGender({
                        ...currentSegmentDirectionAndIndex,
                        data: null,
                    }),
                );
            }

            if (orderPlaces.length > 0 && !isToggleMode) {
                actions.push(
                    addPlacesToOrder({
                        ...currentSegmentDirectionAndIndex,
                        data: placeNumbers,
                    }),
                );
            } else if (
                places.some(place => place.gender === GENDER_TYPE.SINGLE)
            ) {
                this.setState({places, coords});
            } else {
                // Считаем, что при одновременном выборе нескольких мест в одном купе (например Люкс)
                // все места одного пола и совпадают с первым выбранным местом
                actions.push(
                    setPreferredGender({
                        ...currentSegmentDirectionAndIndex,
                        data: places[0].gender,
                    }),
                );
                actions.push(
                    addPlacesToOrder({
                        ...currentSegmentDirectionAndIndex,
                        data: placeNumbers,
                    }),
                );
            }
        }

        this.props.dispatchActions(actions);
    };

    onGenderChange = (): void => {
        const {currentSegmentDirectionAndIndex} = this.props;
        const {places} = this.state;

        if (!places) {
            return;
        }

        this.props.addPlacesToOrder({
            ...currentSegmentDirectionAndIndex,
            data: places.map(place => place.number),
        });

        this.resetPlace();
    };

    /*
     * Метод для получения нод мест с подсчитаными координатами
     */
    onPlaceCoordsCalculated = (
        schemaPlaceNodeMap: PartialRecord<string, ISchemaPlaceNodeMapItem>,
    ): void => {
        const {coach, currentSegmentDirectionAndIndex} = this.props;

        this.schemaPlaceNodeMap = schemaPlaceNodeMap;

        const hasGroupedPlaces = Object.values(schemaPlaceNodeMap).some(
            place => place && place.numbers.length > 1,
        );

        this.props.setSchemeHasGroupedPlaces({
            ...currentSegmentDirectionAndIndex,
            data: {
                index: coach.index,
                hasGroupedPlaces,
            },
        });

        // кейс, когда сначала мы показывали анимированный скелетон схемы и только потом прислали информацию о местах
        // т.е. TransportSchema уже отрендерилась
        if (this.rendered) {
            this.scrollSchemaToFirstSelectablePlace();
        }
    };

    /*
     * Метод скролит схему на мобилах до первого свободного места
     */
    scrollSchemaToFirstSelectablePlace = (): void => {
        const schemaPlaceNodeMap = this.schemaPlaceNodeMap;
        const scrollableContainer = this.scrollableContainer.current;

        if (
            !this.props.deviceType.isMobile ||
            !schemaPlaceNodeMap ||
            !scrollableContainer
        ) {
            return;
        }

        const windowWidth = window.innerWidth;
        const availablePlaces = Object.values(schemaPlaceNodeMap)
            .filter(nodeItem =>
                this.props.orderStep === ORDER_STEP.CONFIRM
                    ? nodeItem?.place.placeState === SELECTED
                    : nodeItem?.place.placeState === AVAILABLE ||
                      nodeItem?.place.placeState === SELECTED,
            )
            .filter(isNotUndefined);
        const closestToLeftPlace = minBy(
            availablePlaces,
            ({coords}) => coords?.left,
        );

        if (!closestToLeftPlace?.coords) {
            return;
        }

        // Если элемент выходит за видимую область экрана, то помещаем его в видимую область
        if (
            closestToLeftPlace.coords.left + closestToLeftPlace.coords.width >
            windowWidth
        ) {
            scrollableContainer.scrollLeft =
                closestToLeftPlace.coords.left -
                windowWidth / 2 +
                closestToLeftPlace.coords.width / 2 +
                SCHEMA_LEFT_PADDING;
        }
    };

    getGenderPopup(): React.ReactNode {
        const {gender, deviceType} = this.props;
        const {places, coords} = this.state;
        const showGenderSelector = !gender && Boolean(places);

        const genderPopup = (
            <GenderPopup
                className={cx('genderPopup')}
                anchor={this._pseudoPlaceRef.current}
                gender={gender}
                visible={showGenderSelector}
                onChange={this.onGenderChange}
                onClose={this.onCloseGenderSelector}
            />
        );

        if (deviceType.isMobile) {
            return genderPopup;
        }

        return (
            <div
                className={cx('pseudoPlace')}
                ref={this._pseudoPlaceRef}
                style={
                    coords && {
                        height: `${coords.height}px`,
                        width: `${coords.width}px`,
                        top: `${coords.top + SCHEMA_TOP_PADDING}px`,
                        left: `${coords.left}px`,
                    }
                }
            >
                {genderPopup}
            </div>
        );
    }

    resetPlace = (): void => {
        this.setState({places: null});
    };

    onSchemaCatchError = (error: Error): void => {
        const {schema, currentSegmentDirectionAndIndex} = this.props;

        this.props.clearSchemeFromCoaches({
            ...currentSegmentDirectionAndIndex,
            data: schema.id,
        });

        this.props.onSchemaCatchError?.();

        logError(
            {
                message:
                    '[YATRAVEL][TRAINS] Ошибка при рендеринге svg схемы вагонов',
                block: 'trainsPlacesViewType',
            },
            error,
        );
    };

    render(): React.ReactNode {
        const {
            className,
            legendClassName,
            deviceType,
            coach,
            coachSchemaRenderingDelay,
            schema,
            schemaPlaceMaskPrefix,
            gender,
            placeMap,
            passengers,
            interactive,
            scrollOffset = 3,
        } = this.props;
        const {places} = this.state;

        const showGenderSelector = !gender && Boolean(places);

        const placeMapContainer = (
            <div className={cx('placeMapContainer')}>
                <SvgSchemaContainer
                    deviceType={deviceType}
                    coachSchemaRenderingDelay={coachSchemaRenderingDelay}
                    coach={coach}
                    gender={gender}
                    schema={schema}
                    schemaPlaceMaskPrefix={schemaPlaceMaskPrefix}
                    placeMap={placeMap}
                    interactive={Boolean(interactive)}
                    showTooltip={!showGenderSelector}
                    passengers={passengers}
                    onPlaceClick={this.onPlaceClick}
                    onPlaceCoordsCalculated={this.onPlaceCoordsCalculated}
                    onCatchError={this.onSchemaCatchError}
                />
            </div>
        );

        return (
            <div
                className={cx('root', className)}
                {...prepareQaAttributes(this.props)}
            >
                {deviceType.isMobile ? (
                    <HorizontalScroller
                        offset={scrollOffset}
                        scrollerRef={this.scrollableContainer}
                    >
                        {placeMapContainer}
                    </HorizontalScroller>
                ) : (
                    placeMapContainer
                )}

                <SvgSchemaLegend
                    className={legendClassName}
                    coach={coach}
                    deviceType={deviceType}
                />

                {this.getGenderPopup()}
            </div>
        );
    }
}

export default connect(mapStateToProps, mapDispatchToProps)(TransportSchema);
