import React from 'react';
import {times} from 'lodash';

import {IWithDeviceType} from 'types/withDeviceType';

import {deviceMods} from 'utilities/stylesUtils';

import Box from 'components/Box/Box';
import Flex from 'components/Flex/Flex';
import Separator from 'components/Separator/Separator';
import TextSkeleton from 'components/Skeletons/TextSkeleton/TextSkeleton';
import BookingLayout from 'components/Layouts/BookingLayout/BookingLayout';
import ConfirmationTimerSkeleton from 'projects/trains/components/TrainsOrderPage/OrderTimer/ConfirmationTimerSkeleton';

import cx from './ConfirmStepSkeleton.scss';

const TrainBlock: React.FC<IWithDeviceType> = ({
    deviceType: {isDesktop, isMobile},
}) => (
    <BookingLayout.Card>
        <Box>
            <TextSkeleton
                className={cx('train-title')}
                withAnimation
                size={isDesktop ? 'xl' : 'l'}
            />

            {isMobile && (
                <Box>
                    <TextSkeleton
                        className={cx('dateInfo')}
                        withAnimation
                        size="m"
                    />

                    <Box above="3" below="1">
                        <Separator />
                    </Box>
                </Box>
            )}

            <Flex
                justifyContent="space-between"
                flexDirection={isDesktop ? 'row' : 'column'}
                between={isDesktop ? 0 : 3}
                nowrap
                below="3"
            >
                <Box above={isDesktop ? 5 : 3} between="3">
                    {isDesktop && (
                        <TextSkeleton
                            className={cx('train-name')}
                            withAnimation
                            size="xl"
                        />
                    )}

                    <TextSkeleton className={cx('train-info')} withAnimation />
                </Box>

                <Box
                    className={cx('train-time-and-stations')}
                    between={isDesktop ? 3 : 0}
                >
                    <Box>
                        <Flex
                            className={cx('invisible')}
                            justifyContent="space-between"
                        >
                            <TextSkeleton className={cx('date')} size="s" />
                            <TextSkeleton className={cx('date')} size="s" />
                        </Flex>

                        <Flex nowrap alignItems="center" between="2" inline>
                            <TextSkeleton
                                className={cx('time')}
                                withAnimation
                                size="xl"
                            />

                            <Separator className={cx('separator')} />

                            <TextSkeleton
                                className={cx('duration')}
                                withAnimation
                            />

                            <Separator className={cx('separator')} />

                            <TextSkeleton
                                className={cx('time')}
                                withAnimation
                                size="xl"
                            />
                        </Flex>
                    </Box>

                    <Flex justifyContent="space-between">
                        <TextSkeleton className={cx('station')} />
                        <TextSkeleton className={cx('station')} />
                    </Flex>
                </Box>
            </Flex>

            <Box className={cx('invisible')} above={3}>
                <TextSkeleton className={cx('badge')} size="m" />
            </Box>
        </Box>
    </BookingLayout.Card>
);

const CoachBlock: React.FC<IWithDeviceType> = ({deviceType: {isDesktop}}) => (
    <BookingLayout.Card>
        <Box between={isDesktop ? 4 : 3}>
            <TextSkeleton
                className={cx('places-title')}
                withAnimation
                size={isDesktop ? 'xl' : 'l'}
            />

            <Box>
                <TextSkeleton className={cx('coach-title')} withAnimation />
                <TextSkeleton className={cx('coach-info')} withAnimation />

                <TextSkeleton className={cx('coach-places')} withAnimation />
            </Box>

            <TextSkeleton className={cx('coach-schema')} withAnimation />
        </Box>
    </BookingLayout.Card>
);

const PassengerBlock: React.FC<IWithDeviceType> = ({
    deviceType: {isDesktop},
}) => (
    <BookingLayout.Card>
        <TextSkeleton
            className={cx('passengers-title')}
            withAnimation
            size={isDesktop ? 'xl' : 'l'}
        />

        <Box above={isDesktop ? 5 : 4} between="4">
            {isDesktop && <Separator />}

            <Flex
                inline={isDesktop}
                justifyContent="space-between"
                flexDirection={isDesktop ? 'row' : 'column'}
                between={isDesktop ? 2 : 4}
            >
                {times(3, index => (
                    <Box
                        className={cx(
                            'passenger-long',
                            index === 2 && 'invisible',
                        )}
                        key={index}
                    >
                        <TextSkeleton
                            className={cx('passenger-label-long')}
                            withAnimation
                            size="s"
                        />

                        <TextSkeleton
                            className={cx(
                                'passenger-value-long',
                                index >= 1 && 'invisible',
                            )}
                            withAnimation
                        />
                    </Box>
                ))}

                {times(2, index => (
                    <Box
                        className={cx('passenger-short', 'invisible')}
                        key={index}
                    >
                        <TextSkeleton
                            className={cx('passenger-label-short')}
                            size="s"
                        />
                        <TextSkeleton className={cx('passenger-value-short')} />
                    </Box>
                ))}
            </Flex>

            <TextSkeleton className={cx('passengers-price', 'invisible')} />
        </Box>
    </BookingLayout.Card>
);

const ContactsBlock: React.FC<IWithDeviceType> = ({
    deviceType: {isDesktop, isMobile},
}) => (
    <BookingLayout.Card>
        <TextSkeleton
            className={cx('contacts-title')}
            withAnimation
            size={isDesktop ? 'xl' : 'l'}
        />

        <Box inline={isDesktop} between={isDesktop ? 8 : 4} above="4">
            {times(2, index => (
                <Box
                    className={cx(isMobile && index === 1 && 'invisible')}
                    key={index}
                >
                    <TextSkeleton
                        className={cx('contacts-label')}
                        withAnimation
                        size="s"
                    />

                    <TextSkeleton
                        className={cx('contacts-value')}
                        withAnimation
                    />
                </Box>
            ))}
        </Box>
    </BookingLayout.Card>
);

const ConfirmStepSkeleton: React.FC<IWithDeviceType> = ({deviceType}) => (
    <BookingLayout.Forms className={cx('root', deviceMods('root', deviceType))}>
        {deviceType.isMobile && (
            <Box below="5">
                <ConfirmationTimerSkeleton deviceType={deviceType} />
            </Box>
        )}

        <Box
            className={cx('headerWrapper')}
            above={deviceType.isDesktop ? 5 : 7}
        >
            <TextSkeleton
                className={cx('header')}
                size={deviceType.isMobile ? 'l' : 'xl'}
                withAnimation
            />

            {deviceType.isMobile && (
                <TextSkeleton className={cx('invisible')} size="l" />
            )}
        </Box>

        <TrainBlock deviceType={deviceType} />
        <CoachBlock deviceType={deviceType} />
        <PassengerBlock deviceType={deviceType} />
        <ContactsBlock deviceType={deviceType} />
    </BookingLayout.Forms>
);

export default ConfirmStepSkeleton;
