import React, {ReactNode, useCallback} from 'react';
import times from 'lodash/times';

import {IWithClassName} from 'types/withClassName';
import {DIRECTIONS, EDirection} from 'types/common/EDirection';
import {TAllSegments} from 'projects/trains/components/TrainsOrderSegments/types';
import {ITrainsDirectionAndIndex} from 'types/trains/common/segment/ITrainsDirectionAndIndex';

import {IOrderTripInfo} from 'selectors/trains/order/orderTripInfoSelector';

import {
    getQa,
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';

import OrderSegment from 'projects/trains/components/TrainsOrderSegments/components/OrderSegment/OrderSegment';
import BookingLayout from 'components/Layouts/BookingLayout/BookingLayout';
import Separator from 'components/Separator/Separator';
import getSegmentPaddings from 'projects/trains/components/TrainsOrderSegments/utiltities/getSegmentPaddings';
import ChangeTrainLink from 'projects/trains/components/TrainsOrderSegments/components/ChangeTrainLink/ChangeTrainLink';
import OrderSegmentsHeader from 'projects/trains/components/TrainsOrderSegments/components/OrderSegmentsHeader/OrderSegmentsHeader';

import TransferSeparator from '../TransferSeparator/TransferSeparator';

import cx from './TrainsOrderSegments.scss';

interface ITrainsOrderSegmentProps extends IWithClassName, IWithQaAttributes {
    segments: TAllSegments;
    tripInfo: IOrderTripInfo | null;
    loading?: boolean;
    activeSegmentDirectionAndIndex?: ITrainsDirectionAndIndex;
    showETicketBadge?: boolean;
    canVisibleCoachInfo?: boolean;
    canVisibleNonRefundableLabel?: boolean;
    canHideCompanyForMobileView?: boolean;
    footerNode?: ReactNode;
    withChangeTrainLink?: boolean;
}

const TrainsOrderSegments: React.FC<ITrainsOrderSegmentProps> = props => {
    const {
        className,
        segments,
        tripInfo,
        loading,
        footerNode,
        showETicketBadge,
        canVisibleCoachInfo,
        canHideCompanyForMobileView,
        canVisibleNonRefundableLabel,
        activeSegmentDirectionAndIndex,
        withChangeTrainLink,
    } = props;
    const activeDirection = activeSegmentDirectionAndIndex?.direction;
    const activeIndex = activeSegmentDirectionAndIndex?.index;
    const deviceType = useDeviceType();

    const rootQa = getQa(props);
    const renderHeaderNode = useCallback(
        (direction: EDirection) => {
            if (!tripInfo) {
                return null;
            }

            return (
                <OrderSegmentsHeader
                    direction={direction}
                    tripInfo={tripInfo}
                    {...prepareQaAttributes(rootQa)}
                />
            );
        },
        [rootQa, tripInfo],
    );

    if (!tripInfo) {
        return null;
    }

    const isRoundTrip = Boolean(tripInfo.returnWhen);

    return (
        <BookingLayout.Card
            className={cx('root', className, deviceMods('root', deviceType))}
            sectioned
        >
            {isRoundTrip && renderHeaderNode(EDirection.FORWARD)}
            {DIRECTIONS.map(direction => {
                const directionTrains = segments[direction];
                const trainsCount = Object.keys(directionTrains).length;

                if (trainsCount === 0) {
                    return null;
                }

                const multipleTrainsInDirection = trainsCount > 1;
                const canUseHeaderSeparator =
                    deviceType.isMobile ||
                    (deviceType.isDesktop && multipleTrainsInDirection);

                return (
                    <div key={direction}>
                        {!isRoundTrip && renderHeaderNode(direction)}
                        {canUseHeaderSeparator && (
                            <Separator className={cx('separator')} />
                        )}
                        {times(trainsCount, n => {
                            const trainSegment = directionTrains[n];
                            const nextTrainSegment = directionTrains[n + 1];

                            if (!trainSegment) {
                                return null;
                            }

                            const isCurrentSegmentActive =
                                direction === activeDirection &&
                                n === activeIndex;

                            return (
                                <React.Fragment key={n}>
                                    <div
                                        className={cx('segment', {
                                            segment_active:
                                                (multipleTrainsInDirection ||
                                                    isRoundTrip) &&
                                                isCurrentSegmentActive,
                                        })}
                                    >
                                        <BookingLayout.Card.Section
                                            {...getSegmentPaddings(
                                                deviceType,
                                                multipleTrainsInDirection,
                                                isRoundTrip,
                                            )}
                                        >
                                            <OrderSegment
                                                className={cx('segmentContent')}
                                                deviceType={deviceType}
                                                segment={trainSegment}
                                                showCities={
                                                    multipleTrainsInDirection
                                                }
                                                canVisibleCoachInfo={
                                                    canVisibleCoachInfo
                                                }
                                                citiesIsLoading={loading}
                                                showETicketBadge={
                                                    showETicketBadge
                                                }
                                                canVisibleNonRefundableLabel={
                                                    canVisibleNonRefundableLabel
                                                }
                                                {...prepareQaAttributes({
                                                    key: `${direction}-${n}`,
                                                    current:
                                                        'trainsOrderSegment',
                                                    parent: props,
                                                })}
                                                canHideCompany={
                                                    canHideCompanyForMobileView
                                                }
                                            />

                                            {withChangeTrainLink &&
                                                !multipleTrainsInDirection && (
                                                    <ChangeTrainLink
                                                        className={cx(
                                                            'changeTrainLink',
                                                        )}
                                                        isRoundTrip={
                                                            isRoundTrip
                                                        }
                                                        direction={direction}
                                                    />
                                                )}
                                        </BookingLayout.Card.Section>
                                    </div>

                                    {nextTrainSegment && (
                                        <>
                                            <Separator
                                                className={cx('separator')}
                                            />

                                            <BookingLayout.Card.Section
                                                paddingTop={2}
                                                paddingBottom={2}
                                            >
                                                <TransferSeparator
                                                    deviceType={deviceType}
                                                    hideSeparator
                                                    loading={loading}
                                                    currentSegment={
                                                        trainSegment
                                                    }
                                                    nextSegment={
                                                        nextTrainSegment
                                                    }
                                                    {...prepareQaAttributes({
                                                        parent: props,
                                                        current: 'transfer',
                                                    })}
                                                />
                                            </BookingLayout.Card.Section>

                                            <Separator
                                                className={cx('separator')}
                                            />
                                        </>
                                    )}
                                </React.Fragment>
                            );
                        })}
                        {withChangeTrainLink && multipleTrainsInDirection && (
                            <BookingLayout.Card.Section
                                paddingTop={0}
                                paddingBottom={5}
                            >
                                <ChangeTrainLink
                                    className={cx('changeTrainLink')}
                                    isRoundTrip={isRoundTrip}
                                    multipleTrainsInDirection
                                    direction={direction}
                                />
                            </BookingLayout.Card.Section>
                        )}
                        {isRoundTrip && direction === EDirection.FORWARD && (
                            <BookingLayout.Card.Section
                                paddingTop={0}
                                paddingBottom={0}
                            >
                                <Separator />
                            </BookingLayout.Card.Section>
                        )}
                    </div>
                );
            })}

            {footerNode}
        </BookingLayout.Card>
    );
};

export default TrainsOrderSegments;
