import {memo, FC, useMemo} from 'react';
import {useSelector} from 'react-redux';

import {IWithClassName} from 'types/withClassName';
import {EDirection} from 'types/common/EDirection';
import {isFilledTrainsSearchContext} from 'reducers/trains/context/types';

import {trainsContextSelector} from 'selectors/trains/trainsContextSelector';

import {trainsURLs} from 'projects/trains/lib/urls';
import {prepareTrainsContextForSearchUrl} from 'projects/trains/lib/urls/getTrainsSearchUrl';
import getText from 'projects/trains/components/TrainsOrderSegments/components/ChangeTrainLink/utilities/getText';

import Link from 'components/Link/Link';

export interface IChangeTrainLinkInfo extends IWithClassName {
    isRoundTrip: boolean;
    direction: EDirection;
    multipleTrainsInDirection?: boolean;
}

interface IChangeTrainLinkProps extends IChangeTrainLinkInfo {}

const ChangeTrainLink: FC<IChangeTrainLinkProps> = props => {
    const {className, isRoundTrip, direction, multipleTrainsInDirection} =
        props;

    const context = useSelector(trainsContextSelector);

    const text = useMemo(
        () =>
            getText({
                isRoundTrip,
                multipleTrainsInDirection,
                direction,
            }),
        [direction, isRoundTrip, multipleTrainsInDirection],
    );

    const url = useMemo(() => {
        if (!isFilledTrainsSearchContext(context)) {
            return null;
        }

        return trainsURLs.getTrainsSearchUrl({
            context: prepareTrainsContextForSearchUrl(context),
            forwardSegmentId: context.forwardSegmentId,
            direction,
        });
    }, [context, direction]);

    if (!url || !text) {
        return null;
    }

    return (
        <Link className={className} to={url}>
            {text}
        </Link>
    );
};

export default memo(ChangeTrainLink);
