import React, {useMemo} from 'react';

import {ITrainSegment} from 'projects/trains/components/TrainsOrderSegments/types';
import {IWithDeviceType} from 'types/withDeviceType';

import {
    HUMAN_DATE_SHORT_WITH_SHORT_WEEKDAY,
    HUMAN_SHORT,
} from 'utilities/dateUtils/formats';
import {
    prepareQaAttributes,
    IWithQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {
    getLocalArrivalTime,
    getLocalDepartureTime,
} from 'projects/trains/lib/segments/cachedSegmentTime';

import * as i18nBlock from 'i18n/trains-timezones';

import TrainsDuration from 'projects/trains/components/TrainSegment/TrainsDuration/TrainsDuration';
import TrainSegmentTime from 'projects/trains/components/TrainSegment/SegmentTime/TrainSegmentTime';

import cx from './OrderSegmentTimeAndDuration.scss';

interface IOrderSegmentTimeAndDurationPropTypes
    extends IWithDeviceType,
        IWithQaAttributes {
    segment: ITrainSegment;
    duration: number;
}

const OrderSegmentTimeAndDuration: React.FC<IOrderSegmentTimeAndDurationPropTypes> =
    props => {
        const {segment, deviceType, duration} = props;

        const arrivalMoment = useMemo(
            () => getLocalArrivalTime(segment),
            [segment],
        );
        const departureMoment = useMemo(
            () => getLocalDepartureTime(segment),
            [segment],
        );

        const dateFormat = deviceType.isMobile
            ? HUMAN_DATE_SHORT_WITH_SHORT_WEEKDAY
            : HUMAN_SHORT;

        const isNextDay = useMemo(() => {
            if (deviceType.isMobile) {
                return false;
            }

            const tomorrowAfterDeparture = departureMoment
                .clone()
                .add(1, 'days');

            return tomorrowAfterDeparture.isSame(arrivalMoment, 'days');
        }, [arrivalMoment, departureMoment, deviceType.isMobile]);

        return (
            <div className={cx('root')}>
                <div
                    className={cx('times', {
                        times_mobile: deviceType.isMobile,
                    })}
                >
                    <div
                        className={cx('timeMessage')}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'timeMessage',
                        })}
                    >
                        {i18nBlock.timeDashLocal()}
                    </div>

                    <div className={cx('timeArrow')} />

                    <TrainSegmentTime
                        className={cx('time', 'time_departure')}
                        dateClassName={cx('timeDate')}
                        timeClassName={cx('timeHours')}
                        showDate={true}
                        timeMoment={departureMoment}
                        dateFormat={dateFormat}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'departure',
                        })}
                    />

                    <TrainsDuration
                        className={cx('duration')}
                        duration={duration}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'duration',
                        })}
                    />

                    <TrainSegmentTime
                        className={cx('time', 'time_arrival')}
                        dateClassName={cx('timeDate')}
                        timeClassName={cx('timeHours')}
                        showDate={true}
                        timeMoment={arrivalMoment}
                        dateFormat={dateFormat}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'arrival',
                        })}
                        isNextDay={isNextDay}
                    />
                </div>
            </div>
        );
    };

export default OrderSegmentTimeAndDuration;
