import React from 'react';

import {TRAIN_FEATURES_TYPES} from 'projects/trains/lib/segments/features/types';

import capitalizeFirstLetter from 'utilities/strings/capitalizeFirstLetter';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/trains-transport-types';

import Flex from 'components/Flex/Flex';
import TrainSegmentInfo from 'projects/trains/components/TrainSegment/SegmentInfo/TrainSegmentInfo';
import ThroughCoachNotification from 'projects/trains/components/TrainsOrderPage/ThroughCoachNotification/ThroughCoachNotification';
import OrderSegmentTimeAndStations from 'projects/trains/components/TrainsOrderSegments/components/OrderSegment/components/OrderSegmentTimeAndStations/OrderSegmentTimeAndStations';
import Text from 'components/Text/Text';
import OrderSegmentETicketInfo from 'projects/trains/components/TrainsOrderSegments/components/OrderSegment/components/OrderSegmentETicketInfo/OrderSegmentETicketInfo';
import {IOrderSegmentProps} from 'projects/trains/components/TrainsOrderSegments/components/OrderSegment/OrderSegment';

import OrderSegmentCoachInfo from '../OrderSegmentCoachInfo/OrderSegmentCoachInfo';
import OrderSegmentNonRefundableLabel from '../OrderSegmentNonRefundableLabel/OrderSegmentNonRefundableLabel';

import cx from './OrderSegmentDesktop.scss';

interface IOrderSegmentDesktopProps extends IOrderSegmentProps {
    duration: number;
}

const OrderSegmentDesktop: React.FC<IOrderSegmentDesktopProps> = props => {
    const {
        className,
        segment,
        deviceType,
        duration,
        showCities,
        citiesIsLoading,
        showETicketBadge,
        canVisibleCoachInfo,
        canVisibleNonRefundableLabel,
    } = props;
    const {
        title,
        arrival,
        features,
        coachInfo,
        stationTo,
        originalArrival,
        hasNonRefundableTariff,
        isThroughCoachArrival,
    } = segment;
    const companyTrainFeatures = features.filter(
        feature => feature.type === TRAIN_FEATURES_TYPES.COMPANY,
    );
    const trainsFeaturesAfterTransportInfo = features.filter(feature => {
        return ![
            TRAIN_FEATURES_TYPES.ETICKET,
            TRAIN_FEATURES_TYPES.DYNAMIC_PRICING,
            TRAIN_FEATURES_TYPES.COMPANY,
        ].includes(feature.type);
    });

    return (
        <div className={className} {...prepareQaAttributes(props)}>
            <div className={cx('section')}>
                {companyTrainFeatures.length > 0 && (
                    <div className={cx('transportInfo')}>
                        <TrainSegmentInfo
                            trainFeatures={companyTrainFeatures}
                            deviceType={deviceType}
                            {...prepareQaAttributes(props)}
                            size="xl"
                        />

                        <Text
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'numberAndDirection',
                            })}
                            tag="div"
                            className={cx('numberAndDirectionOffset')}
                        >
                            {`${capitalizeFirstLetter(
                                i18nBlock.typeDashTrain(),
                            )} ${segment.number || ''}, ${title}`}
                        </Text>

                        <TrainSegmentInfo
                            trainFeatures={trainsFeaturesAfterTransportInfo}
                            deviceType={deviceType}
                            size="m"
                            {...prepareQaAttributes(props)}
                        />
                        {canVisibleCoachInfo && coachInfo && (
                            <OrderSegmentCoachInfo
                                coachInfo={coachInfo}
                                {...prepareQaAttributes(props)}
                            />
                        )}
                    </div>
                )}
                <div className={cx('timeAndStations')}>
                    <OrderSegmentTimeAndStations
                        segment={segment}
                        duration={duration}
                        deviceType={deviceType}
                        showCities={showCities}
                        citiesIsLoading={citiesIsLoading}
                        {...prepareQaAttributes(props)}
                    />
                </div>
            </div>
            <ThroughCoachNotification
                className={cx('throughCoachNotification')}
                arrival={arrival}
                stationTo={stationTo}
                originalArrival={originalArrival}
                isThroughCoachArrival={isThroughCoachArrival}
            />
            <Flex inline above={3} between={2} alignItems="center">
                {canVisibleNonRefundableLabel && hasNonRefundableTariff && (
                    <OrderSegmentNonRefundableLabel />
                )}
                {showETicketBadge && (
                    <OrderSegmentETicketInfo
                        eTicketStatus={segment.eTicketStatus}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'electronicRegistration',
                        })}
                    />
                )}
            </Flex>
        </div>
    );
};

export default OrderSegmentDesktop;
