import React, {useMemo} from 'react';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {EDirection} from 'types/common/EDirection';

import {IOrderTripInfo} from 'selectors/trains/order/orderTripInfoSelector';

import {
    ROBOT_DATETIME_UTC,
    HUMAN_DATE_WITH_FULL_WEEKDAY,
    HUMAN_SHORT,
} from 'utilities/dateUtils/formats';
import {CHAR_EM_DASH, CHAR_NBSP} from 'utilities/strings/charCodes';
import {formatDate} from 'utilities/dateUtils';
import {deviceMods} from 'utilities/stylesUtils';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18n from 'i18n/trainsOrderSegments';

import Text from 'components/Text/Text';

import cx from './OrderSegmentsTitle.scss';

interface IOrderSegmentsTitleProps
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    tripInfo: IOrderTripInfo;
    direction: EDirection;
}

const OrderSegmentsTitle: React.FC<IOrderSegmentsTitleProps> = props => {
    const {
        className,
        tripInfo: {from, to, when, returnWhen},
        deviceType,
        direction,
    } = props;

    const isForward = direction === EDirection.FORWARD;
    const points = isForward ? [from, to] : [to, from];

    const timeDescription = useMemo(() => {
        if (returnWhen) {
            return (
                <span className={cx('timeDescription')}>
                    {`${i18n.forwardAndBackward()}, `}

                    <time
                        dateTime={formatDate(when, ROBOT_DATETIME_UTC)}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'departure',
                        })}
                    >
                        {formatDate(when, HUMAN_SHORT).toLowerCase()}
                    </time>

                    {`${CHAR_NBSP}${CHAR_EM_DASH} `}

                    <time
                        dateTime={formatDate(returnWhen, ROBOT_DATETIME_UTC)}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'arrival',
                        })}
                    >
                        {formatDate(returnWhen, HUMAN_SHORT).toLowerCase()}
                    </time>
                </span>
            );
        }

        return (
            <time
                className={cx('timeDescription')}
                dateTime={formatDate(when, ROBOT_DATETIME_UTC)}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'departure',
                })}
            >
                {formatDate(when, HUMAN_DATE_WITH_FULL_WEEKDAY).toLowerCase()}
            </time>
        );
    }, [props, returnWhen, when]);

    return (
        <Text
            tag="h1"
            size={deviceType.isMobile ? 'm' : 'xl'}
            className={cx('root', className, deviceMods('root', deviceType))}
            {...prepareQaAttributes(props)}
        >
            <span
                className={cx('direction')}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'direction',
                })}
            >
                {points.join(`${CHAR_NBSP}${CHAR_EM_DASH} `)}
            </span>

            {timeDescription}
        </Text>
    );
};

export default OrderSegmentsTitle;
