import React, {Component} from 'react';
import {connect} from 'react-redux';
import {RouteComponentProps, withRouter} from 'react-router-dom';

import {EProjectName} from 'constants/common';

import {EFooterProject} from 'components/Footer/types';
import {EDirection} from 'types/common/EDirection';
import {IWithDeviceType} from 'types/withDeviceType';
import {ETrainsGoal} from 'utilities/metrika/types/goals/trains';

import {StoreInterface} from 'reducers/storeTypes';
import trainsFetchOrderSegment from 'reducers/trains/order/thunk/trainsFetchOrderSegment';

import deviceTypeSelector from 'selectors/common/deviceTypeSelector';
import trainsOrderSelector from 'selectors/trains/order/trainsOrderSelector';

import {reachGoal} from 'utilities/metrika';
import {trainsURLs} from 'projects/trains/lib/urls';
import {getTrainsOrderParamsByQuery} from 'projects/trains/lib/urls/getTrainsOrderParamsByQuery/getTrainsOrderParamsByQuery';
import getQueryByLocation from 'utilities/getQueryByLocation/getQueryByLocation';
import getFirstForwardSegment from 'projects/trains/lib/complexOrder/getFirstForwardSegment';

import * as i18nBlock from 'i18n/trains-partner-redirect';

import Spinner from 'components/Spinner/Spinner';
import LayoutDefault from 'components/Layouts/LayoutDefault/LayoutDefault';

import {serverFetchDataDispatcher} from 'contexts/ServerFetchDataContext';

import {fetchOrderData} from 'server/redux/trains/order';

import cx from './styles.scss';

const TICK_INTERVAL = 1000;

interface ITrainsPartnerRedirectPageStateProps extends IWithDeviceType {
    url: string;
}

const mapStateToProps = (
    state: StoreInterface,
): ITrainsPartnerRedirectPageStateProps => {
    const order = trainsOrderSelector(state);
    const segment = getFirstForwardSegment(order);
    const deviceType = deviceTypeSelector(state);

    if (!segment) {
        return {
            url: '#',
            deviceType,
        };
    }

    return {
        url: trainsURLs.getUfsOrderUrl(segment),
        deviceType,
    };
};

const mapDispatchToProps = {
    fetchOrderSegment: trainsFetchOrderSegment,
};

type TTrainsPartnerRedirectPageDispatchProps = typeof mapDispatchToProps;

interface ITrainsPartnerRedirectPagePropTypes
    extends ITrainsPartnerRedirectPageStateProps,
        TTrainsPartnerRedirectPageDispatchProps,
        RouteComponentProps {}

interface ITrainsPartnerRedirectPageState {
    counter: number;
}

class TrainsPartnerRedirectPage extends Component<
    ITrainsPartnerRedirectPagePropTypes,
    ITrainsPartnerRedirectPageState
> {
    state: ITrainsPartnerRedirectPageState = {
        counter: 5,
    };

    private intervalId: number | null = null;

    componentDidMount(): void {
        const {url, fetchOrderSegment, location} = this.props;

        reachGoal(ETrainsGoal.PARTNER_REDIRECT_LOAD);

        const query = getTrainsOrderParamsByQuery(getQueryByLocation(location));
        const route = query[EDirection.FORWARD]?.[0];

        if (route) {
            fetchOrderSegment(route, EDirection.FORWARD, 0);
        }

        this.intervalId = window.setInterval(() => {
            const {counter} = this.state;

            if (counter === 1 && this.intervalId) {
                window.clearInterval(this.intervalId);
                window.location.href = url;
            } else {
                this.setState({
                    counter: counter - 1,
                });
            }
        }, TICK_INTERVAL);
    }

    componentWillUnmount(): void {
        if (this.intervalId) {
            window.clearInterval(this.intervalId);
        }
    }

    render(): React.ReactNode {
        const {url, deviceType} = this.props;

        return (
            <LayoutDefault
                isFixedNavigation
                showSearchForm
                showNavigation
                project={EProjectName.TRAINS}
                footerType={EFooterProject.TRAINS}
            >
                <div className={cx('root', {root_mobile: deviceType.isMobile})}>
                    <Spinner className={cx('spinner')} />

                    <div className={cx('counter')}>{this.state.counter}</div>

                    <div className={cx('description')}>
                        <h1 className={cx('title')}>{i18nBlock.title()}</h1>

                        <div className={cx('directLinkHolder')}>
                            <a
                                href={url}
                                className={cx('directLink')}
                                rel="nofollow"
                            >
                                {i18nBlock.directDashLink()}
                            </a>
                        </div>
                    </div>
                </div>
            </LayoutDefault>
        );
    }
}

export default withRouter(
    serverFetchDataDispatcher([fetchOrderData])(
        connect(mapStateToProps, mapDispatchToProps)(TrainsPartnerRedirectPage),
    ),
);
