import React, {Component} from 'react';

import {
    TTrainPopularRoutes,
    ITrainPopularRoutesItemLink,
} from 'types/trains/common/popularRoutes/popularRoutes';

import {IDevice} from 'reducers/common/commonReducerTypes';

import groups from 'utilities/array/groups';
import {trainsURLs} from 'projects/trains/lib/urls';

import Link from 'components/Link/Link';

import cx from './styles.scss';

interface ITrainsPopularRoutesListProps {
    activeTab: string;
    popularRoutes: TTrainPopularRoutes;
    deviceType: IDevice;
}

export default class TrainsPopularRoutesList extends Component<ITrainsPopularRoutesListProps> {
    private renderItem(
        item: ITrainPopularRoutesItemLink,
        code: string,
    ): React.ReactNode {
        const {from, to, fromSlug, toSlug} = item;

        return (
            <div key={`${code}-${from}-${to}`} className={cx('item')}>
                <Link
                    url={trainsURLs.getTrainsSearchUrl({
                        context: {
                            from: fromSlug,
                            to: toSlug,
                        },
                    })}
                >
                    {from} — {to}
                </Link>
            </div>
        );
    }

    private renderMobile(): React.ReactNode {
        const {activeTab, popularRoutes} = this.props;

        return (
            <div className={cx('root', 'root_mobile')}>
                {popularRoutes.map(routeItem => (
                    <section
                        key={routeItem.code}
                        className={cx('type')}
                        aria-hidden={routeItem.code !== activeTab}
                    >
                        {routeItem.links.map(item =>
                            this.renderItem(item, routeItem.code),
                        )}
                    </section>
                ))}
            </div>
        );
    }

    private renderDesktop(): React.ReactNode {
        const {activeTab, popularRoutes} = this.props;

        return (
            <div className={cx('root', 'root_desktop')}>
                {popularRoutes.map(routeItem => (
                    <section
                        key={routeItem.code}
                        className={cx('type')}
                        aria-hidden={routeItem.code !== activeTab}
                    >
                        {groups(routeItem.links, 3).map((group, index) => (
                            <div key={index} className={cx('group')}>
                                {group.map(item =>
                                    this.renderItem(item, routeItem.code),
                                )}
                            </div>
                        ))}
                    </section>
                ))}
            </div>
        );
    }

    render(): React.ReactNode {
        const {deviceType} = this.props;

        if (deviceType.isMobile) {
            return this.renderMobile();
        }

        return this.renderDesktop();
    }
}
