import React, {Component, Fragment} from 'react';
import {connect} from 'react-redux';
import {RouteComponentProps, withRouter} from 'react-router-dom';

import {EProjectName} from 'constants/common';

import {ServerDataFetcherBag} from 'server/redux/types';
import {
    ETrainPopularRoutesState,
    ITrainPopularRoutesWithState,
} from 'reducers/trains/popularRoutes/types';
import {EFooterProject} from 'components/Footer/types';

import {StoreInterface} from 'reducers/storeTypes';
import {IDevice} from 'reducers/common/commonReducerTypes';
import {CustomDispatch} from 'reducers/trains/customDispatch';
import {trainsFetchPopularRoutes} from 'reducers/trains/popularRoutes/actions';

import {deviceMods} from 'utilities/stylesUtils';

import * as trainsPopularRoutesKeyset from 'i18n/trains-popular-routes';

import Spinner from 'components/Spinner/Spinner';
import LayoutLight from 'components/Layouts/LayoutLight/LayoutLight';
import TrainsPopularRoutesTabs from 'projects/trains/components/TrainsPopularRoutesPage/TrainsPopularRoutesTabs';
import TrainsPopularRoutesList from 'projects/trains/components/TrainsPopularRoutesPage/TrainsPopularRoutesList';
import TrainsPopularRoutesBreadcrumbs from 'projects/trains/components/TrainsPopularRoutesPage/TrainsPopularRoutesBreadcrumbs';
import SearchForm from 'projects/trains/components/SearchForm/SearchForm';

import DeviceTypeContext from 'contexts/DeviceTypeContext';
import {serverFetchDataDispatcher} from 'contexts/ServerFetchDataContext';

import trainsServerFetchPopularRoutes from 'server/redux/trains/trainsServerFetchPopularRoutes';

import cx from './styles.scss';

interface ITrainsPopularRoutesPageProps extends RouteComponentProps {
    popularRoutes: ITrainPopularRoutesWithState;
    fetchPopularRoutes: () => void;
}

interface ITrainsPopularRoutesPageState {
    activeTab: string | null;
    prevPopularRoutesState: ETrainPopularRoutesState;
}

const DEFAULT_ACTIVE_TAB = 'moscow';

class TrainsPopularRoutesPage extends Component<
    ITrainsPopularRoutesPageProps,
    ITrainsPopularRoutesPageState
> {
    state: ITrainsPopularRoutesPageState = {
        activeTab: null,
        prevPopularRoutesState: ETrainPopularRoutesState.INITIAL,
    };

    static getDerivedStateFromProps(
        nextProps: ITrainsPopularRoutesPageProps,
        prevState: ITrainsPopularRoutesPageState,
    ) {
        const {popularRoutes} = nextProps;

        if (
            popularRoutes.state === ETrainPopularRoutesState.RESOLVE &&
            prevState.prevPopularRoutesState !==
                ETrainPopularRoutesState.RESOLVE
        ) {
            return {
                activeTab: popularRoutes.items.find(
                    item => item.code === DEFAULT_ACTIVE_TAB,
                )
                    ? DEFAULT_ACTIVE_TAB
                    : popularRoutes.items[0].code,
                prevPopularRoutesState: popularRoutes.state,
            };
        }

        return null;
    }

    componentDidMount(): void {
        const {popularRoutes, fetchPopularRoutes} = this.props;

        if (popularRoutes.state === ETrainPopularRoutesState.INITIAL) {
            fetchPopularRoutes();
        }
    }

    private handleTabChange = (code: string): void => {
        this.setState({activeTab: code});
    };

    private renderContent(deviceType: IDevice): React.ReactNode {
        const {activeTab} = this.state;
        const {popularRoutes} = this.props;

        if (popularRoutes.state === ETrainPopularRoutesState.PENDING) {
            return (
                <div className={cx('loading')}>
                    <Spinner size="xl" />
                </div>
            );
        }

        if (activeTab === null) {
            return null;
        }

        return (
            <Fragment>
                <TrainsPopularRoutesTabs
                    popularRoutes={popularRoutes.items}
                    activeTab={activeTab}
                    deviceType={deviceType}
                    onChange={this.handleTabChange}
                />

                <TrainsPopularRoutesList
                    popularRoutes={popularRoutes.items}
                    activeTab={activeTab}
                    deviceType={deviceType}
                />
            </Fragment>
        );
    }

    render(): React.ReactNode {
        return (
            <DeviceTypeContext.Consumer>
                {deviceType => (
                    <LayoutLight
                        className={cx('root', deviceMods('root', deviceType))}
                        project={EProjectName.TRAINS}
                        footerType={EFooterProject.TRAINS}
                    >
                        <div
                            className={cx(
                                'wrapper',
                                deviceMods('wrapper', deviceType),
                            )}
                        >
                            <SearchForm className={cx('searchForm')} />

                            <h1 className={cx('title')}>
                                {trainsPopularRoutesKeyset.title()}
                            </h1>

                            <div className={cx('description')}>
                                {trainsPopularRoutesKeyset.description()}
                            </div>

                            {this.renderContent(deviceType)}

                            <TrainsPopularRoutesBreadcrumbs />
                        </div>
                    </LayoutLight>
                )}
            </DeviceTypeContext.Consumer>
        );
    }
}

async function prefetchData(serverFetcherData: ServerDataFetcherBag) {
    await trainsServerFetchPopularRoutes(serverFetcherData);
}

const mapStateToProps = (state: StoreInterface) => ({
    popularRoutes: state.trains.popularRoutes,
});

const mapDispatchToProps = (dispatch: CustomDispatch) => ({
    fetchPopularRoutes: () => dispatch(trainsFetchPopularRoutes()),
});

export default withRouter(
    connect(
        mapStateToProps,
        mapDispatchToProps,
    )(serverFetchDataDispatcher([prefetchData])(TrainsPopularRoutesPage)),
);
