import React from 'react';
import {useSelector} from 'react-redux';

import {isFilledTrainsSearchContext} from 'reducers/trains/context/types';
import {
    TTrainsOrderStep,
    TTrainsSearchStep,
} from 'types/trains/common/steps/ITrainsSearchAndOrderSteps';
import {IWithClassName} from 'types/withClassName';

import {trainsContextSelector} from 'selectors/trains/trainsContextSelector';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import getTrainsStepKey from './utilities/getTrainsStepKey';
import getTrainsOrderStepText from './utilities/getTrainsOrderStepText';
import getTrainsSearchStepUrl from './utilities/getTrainsSearchStepUrl';
import getTrainsSearchStepText from './utilities/getTrainsSearchStepText';

import {TBoxSizes} from 'components/Box/Box';
import Breadcrumbs from 'components/Breadcrumbs/Breadcrumbs';
import Container from 'components/Layouts/Container/Container';

import cx from './TrainsSearchAndOrderSteps.scss';

interface ITrainsSearchAndOrderStepsProps
    extends IWithClassName,
        IWithQaAttributes {
    orderSteps: TTrainsOrderStep[];
    searchSteps: TTrainsSearchStep[];
    x?: TBoxSizes;
    breadcrumbsClassName?: string;
    handleChangeOrderStep?: (step: TTrainsOrderStep) => void;
}

const TrainsSearchAndOrderSteps: React.FC<ITrainsSearchAndOrderStepsProps> =
    props => {
        const {
            x,
            className,
            orderSteps,
            searchSteps,
            breadcrumbsClassName,
            handleChangeOrderStep,
        } = props;
        const context = useSelector(trainsContextSelector);

        if (!isFilledTrainsSearchContext(context)) {
            return null;
        }

        return (
            <Container className={className}>
                <Breadcrumbs
                    x={x}
                    hasBorder
                    className={breadcrumbsClassName}
                    {...prepareQaAttributes(props)}
                >
                    {searchSteps.map(step => {
                        const key = getTrainsStepKey(step);

                        return (
                            <Breadcrumbs.Item
                                className={cx('item')}
                                key={step.type}
                                active={step.isActive}
                                disabled={step.disabled}
                                to={getTrainsSearchStepUrl({
                                    step,
                                    context,
                                })}
                                {...prepareQaAttributes({
                                    parent: props,
                                    current: 'search',
                                    key,
                                })}
                            >
                                {getTrainsSearchStepText({step})}
                            </Breadcrumbs.Item>
                        );
                    })}
                    {orderSteps.map(step => {
                        const key = getTrainsStepKey(step);
                        const handleClickOrderCrumb = (): void => {
                            handleChangeOrderStep?.(step);
                        };

                        return (
                            <Breadcrumbs.Item
                                className={cx('item')}
                                key={key}
                                active={step.isActive}
                                disabled={step.disabled}
                                onClick={handleClickOrderCrumb}
                                {...prepareQaAttributes({
                                    parent: props,
                                    current: 'orderStep',
                                    key,
                                })}
                            >
                                {getTrainsOrderStepText({
                                    step,
                                    isRoundTrip: Boolean(context.returnWhen),
                                })}
                            </Breadcrumbs.Item>
                        );
                    })}
                </Breadcrumbs>
            </Container>
        );
    };

export default TrainsSearchAndOrderSteps;
