import React, {memo} from 'react';
import {useSelector} from 'react-redux';

import {
    TTrainsVariantId,
    TTrainsVariantWithVisibleStatusById,
} from 'types/trains/common/variant/ITrainsVariant';
import {IWithClassName} from 'types/withClassName';
import {ITrainsSearchSort} from 'projects/trains/lib/sort/types';
import {EAdFoxBannerPosition, EAdFoxBannerType} from 'types/AdFox';
import {ITrainsFilledSearchContext} from 'reducers/trains/context/types';
import {ESubscriptionSource} from 'types/subscription/ESubscriptionSource';
import {EQueryingStatus} from 'types/trains/search/searchInfo/ITrainsSearchInfo';
import {ESubscriptionVerticalName} from 'types/subscription/ESubscriptionVerticalName';
import {ITrainsSearchBadgesInfo} from 'types/trains/search/badgesInfo/ITrainsSearchBadgesInfo';
import {ESearchListItemType} from 'projects/trains/components/TrainsSearchVariants/types/ITrainsSearchVariants';

import {trainsCrossSaleEnabledSelector} from 'selectors/trains/trainsCrossSaleEnabledSelector';
import {trainsCleanSerpEnabledSelector} from 'selectors/trains/genericSearch/search/trainsCleanSerpEnabledSelector';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {checkHasTrainsVariantTransfer} from 'projects/trains/lib/genericSearch/variants/checkHasTrainsVariantTransfer';
import {prepareTrainsSearchList} from 'projects/trains/components/TrainsSearchVariants/utilities/prepareTrainsSearchList';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import getVisibleItems from 'projects/trains/components/TrainsSearchVariants/utilities/getVisibleItems';

import OptionalSubscription from 'containers/Subscriptions/Inline/OptionalSubscription';

import Box from 'components/Box/Box';
import AdFoxBanner from 'components/AdFoxBanner/AdFoxBanner';
import TrainsSearchVariant from './components/TrainsSearchVariant/TrainsSearchVariant';
import TrainsSearchVariantWithTransfer from './components/TrainsSearchVariantWithTransfer/TrainsSearchVariantWithTransfer';
import TrainsSearchVariantsDateSeparator from './components/TrainsSearchVariantsDateSeparator/TrainsSearchVariantsDateSeparator';
import useAdfoxBannerKeyRef from 'projects/trains/components/TrainsSearchVariants/hooks/useAdfoxBannerKeyRef';
import HotelsCrossSaleMap from 'projects/trains/components/TrainsSearchVariants/components/HotelsCrossSaleMap/HotelsCrossSaleMap';

import cx from './TrainsSearchVariants.scss';

interface ITrainsSearchVariantsProps extends IWithClassName {
    bannerIndexPosition: number;
    sort: ITrainsSearchSort;
    searchStatus: EQueryingStatus;
    variantIds: TTrainsVariantId[];
    context: ITrainsFilledSearchContext;
    badgesInfo: ITrainsSearchBadgesInfo;
    variantWithVisibleStatusById: TTrainsVariantWithVisibleStatusById;
    visibleItemsCount?: number;
}

const TrainsSearchVariants: React.FC<ITrainsSearchVariantsProps> = props => {
    const {
        className,
        context,
        sort,
        badgesInfo,
        variantIds,
        searchStatus,
        bannerIndexPosition,
        variantWithVisibleStatusById,
        visibleItemsCount,
    } = props;
    const deviceType = useDeviceType();
    const trainsCrossSaleEnabled = useSelector(trainsCrossSaleEnabledSelector);
    const trainsCleanSerpEnabled = useSelector(trainsCleanSerpEnabledSelector);
    const searchList = prepareTrainsSearchList({
        sort,
        context,
        variantIds,
        bannerIndexPosition,
        variantWithVisibleStatusById,
        trainsCrossSaleEnabled,
        trainsCleanSerpEnabled,
    });

    const adfoxBannerKeyRef = useAdfoxBannerKeyRef(
        variantIds,
        bannerIndexPosition,
    );

    const visibleSearchList = visibleItemsCount
        ? getVisibleItems(searchList, visibleItemsCount)
        : searchList;

    return (
        <div className={className}>
            <Box between={3}>
                {visibleSearchList.map(searchListItem => {
                    switch (searchListItem.type) {
                        case ESearchListItemType.VARIANT: {
                            const {variantId} = searchListItem;

                            const {variant, isVisible} =
                                variantWithVisibleStatusById[variantId];

                            if (!variant || !context.direction) {
                                return null;
                            }

                            const hasVariantTransfer =
                                checkHasTrainsVariantTransfer({variant});
                            const TrainsSearchVariantComponent =
                                hasVariantTransfer
                                    ? TrainsSearchVariantWithTransfer
                                    : TrainsSearchVariant;
                            const variantQa = prepareQaAttributes({
                                key: variant.id,
                                current: 'searchVariant',
                            });

                            return (
                                <div
                                    key={variant.id}
                                    className={cx({
                                        variant_isHidden: !isVisible,
                                    })}
                                    {...prepareQaAttributes(variantQa)}
                                >
                                    <TrainsSearchVariantComponent
                                        variant={variant}
                                        context={context}
                                        deviceType={deviceType}
                                        badgesInfo={badgesInfo}
                                        searchStatus={searchStatus}
                                        direction={context.direction}
                                        {...prepareQaAttributes(variantQa)}
                                    />
                                </div>
                            );
                        }

                        case ESearchListItemType.BANNER: {
                            return (
                                <AdFoxBanner
                                    key={adfoxBannerKeyRef.current}
                                    type={EAdFoxBannerType.Inline}
                                    position={EAdFoxBannerPosition.Center}
                                />
                            );
                        }

                        case ESearchListItemType.SUBSCRIPTION_FORM: {
                            return (
                                <OptionalSubscription
                                    key="subscription"
                                    vertical={ESubscriptionVerticalName.Trains}
                                    source={ESubscriptionSource.SEARCH}
                                    type="wide"
                                />
                            );
                        }

                        case ESearchListItemType.HOTELS_CROSS_SALE: {
                            return (
                                <HotelsCrossSaleMap
                                    key="crossSale"
                                    {...prepareQaAttributes(
                                        'hotelsCrossSaleMap',
                                    )}
                                />
                            );
                        }

                        case ESearchListItemType.DAY_SEPARATOR: {
                            const {date} = searchListItem;

                            return (
                                <TrainsSearchVariantsDateSeparator
                                    date={date}
                                    deviceType={deviceType}
                                    key="trainsSearchVariantsDateSeparator"
                                    {...prepareQaAttributes(
                                        'variantsDateSeparator',
                                    )}
                                />
                            );
                        }
                    }

                    return null;
                })}
            </Box>
        </div>
    );
};

export default memo(TrainsSearchVariants);
