import React from 'react';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {getTrainRangeArrivalByFeatures} from 'projects/trains/lib/genericSearch/train/getTrainRangeArrivalByFeatures';
import {checkHasTrainsSegmentTariffClasses} from 'projects/trains/lib/genericSearch/tariffs/checkHasTrainsSegmentTariffClasses';
import {getTrainsSegmentTariffViewType} from 'projects/trains/components/TrainsSearchVariants/components/TrainsSearchSegment/utilities/getTrainsSegmentTariffViewType';

import Card from 'components/Card/Card';
import Flex from 'components/Flex/Flex';
import TrainsSegmentInfo from '../components/TrainsSegmentInfo/TrainsSegmentInfo';
import TrainsSegmentTariffs from '../components/TrainsSegmentTariffs/TrainsSegmentTariffs';
import TrainsSegmentTimings from '../components/TrainsSegmentTimings/TrainsSegmentTimings';
import TrainsSegmentStations from '../components/TrainsSegmentStations/TrainsSegmentStations';

import TrainsSearchVariantBadges from '../../TrainsSearchVariantBadges/TrainsSearchVariantBadges';
import TrainsSearchVariantPriceAndAction from '../../TrainsSearchVariantPriceAndAction/TrainsSearchVariantPriceAndAction';
import {ITrainsSearchSegmentProps} from '../TrainsSearchSegment';

import cx from './TrainsMobileSearchSegment.scss';

const TrainsMobileSearchSegment: React.FC<ITrainsSearchSegmentProps> =
    props => {
        const {
            segment,
            variant,
            className,
            context,
            direction,
            searchStatus,
            variantBadges,
            isPinnedSegment,
            isTransferSegment,
            canVisibleSegmentAction = true,
        } = props;
        const deviceType = useDeviceType();
        const trainRangeArrival = getTrainRangeArrivalByFeatures(
            segment.features,
        );
        // todo move badges to variant component layer
        const canRenderVariantBadges =
            !isTransferSegment && Boolean(variantBadges?.length);
        const tariffViewType = getTrainsSegmentTariffViewType({
            context,
            isTransferSegment,
        });

        return (
            <Card
                shadow="default"
                className={cx('root', className, {
                    root_hasBadges: canRenderVariantBadges,
                })}
                {...prepareQaAttributes(props)}
            >
                <Flex flexDirection="column">
                    <TrainsSegmentTimings
                        arrival={segment.arrival}
                        departure={segment.departure}
                        stationFrom={segment.stationFrom}
                        stationTo={segment.stationTo}
                        duration={segment.duration}
                        trainRangeArrival={trainRangeArrival}
                        deviceType={deviceType}
                        {...prepareQaAttributes(props)}
                    />
                    <TrainsSegmentStations
                        className={cx('stations')}
                        stationFrom={segment.stationFrom}
                        stationTo={segment.stationTo}
                        {...prepareQaAttributes(props)}
                    />
                    <TrainsSegmentInfo
                        className={cx('info')}
                        segment={segment}
                        deviceType={deviceType}
                        size="s"
                        {...prepareQaAttributes(props)}
                    />
                    <div
                        className={cx('footer', {
                            footer_hasTariffs:
                                checkHasTrainsSegmentTariffClasses(segment),
                        })}
                    >
                        <TrainsSegmentTariffs
                            tariffs={segment.tariffs}
                            train={segment.train}
                            features={segment.features}
                            variant={variant}
                            context={context}
                            tariffViewType={tariffViewType}
                            deviceType={deviceType}
                            isPinnedSegment={isPinnedSegment}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'tariffs',
                            })}
                        />
                        {canVisibleSegmentAction && (
                            <TrainsSearchVariantPriceAndAction
                                className={cx('priceAndAction')}
                                context={context}
                                variant={variant}
                                direction={direction}
                                searchStatus={searchStatus}
                                deviceType={deviceType}
                                isPinnedSegment={isPinnedSegment}
                                canHidePrice
                                {...prepareQaAttributes(props)}
                            />
                        )}
                    </div>
                </Flex>
                {canRenderVariantBadges && (
                    <TrainsSearchVariantBadges variantBadges={variantBadges} />
                )}
            </Card>
        );
    };

export default TrainsMobileSearchSegment;
