import React, {ReactNode, useMemo} from 'react';

import {ITrainsSegment} from 'types/trains/common/segment/ITrainsSegment';
import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {ETrainsFeatureType} from 'types/trains/common/features/ETrainsFeatureType';

import {CHAR_SPACE} from 'utilities/strings/charCodes';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {getTrainNameByFeatures} from 'projects/trains/lib/genericSearch/train/getTrainNameByFeatures';

import * as i18nBlock from 'i18n/trains-segment';

import Text, {TTextSize} from 'components/Text/Text';
import DotSeparator from 'components/DotSeparator/DotSeparator';

interface ITrainsSegmentInfoProps
    extends IWithQaAttributes,
        IWithClassName,
        IWithDeviceType {
    segment: ITrainsSegment;
    size?: TTextSize;
}

const TrainsSegmentInfo: React.FC<ITrainsSegmentInfoProps> = props => {
    const {className, segment, size = 'm'} = props;
    const {train, features, company} = segment;

    const trainNumberNode = useMemo<ReactNode>(() => {
        if (!train?.displayNumber) {
            return null;
        }

        return (
            <Text
                size={size}
                color="primary"
                {...prepareQaAttributes({
                    parent: props,
                    current: 'number',
                })}
            >
                {train.displayNumber}
            </Text>
        );
    }, [train, size]);

    const throughTrainFeatureNode = useMemo<ReactNode>(() => {
        if (!features?.[ETrainsFeatureType.THROUGH_TRAIN]) {
            return null;
        }

        return (
            <Text size={size} color="primary">
                <DotSeparator />
                <span>{i18nBlock.throughDashTrain()}</span>
            </Text>
        );
    }, [features, size]);

    const companyNode = useMemo<ReactNode>(() => {
        if (!company?.title) {
            return null;
        }

        return (
            <Text size={size} color="secondary">
                <DotSeparator />
                <span
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'company',
                    })}
                >
                    {company.title}
                </span>
            </Text>
        );
    }, [company, size]);

    const subTypeFeatureNode = useMemo<ReactNode>(() => {
        const subTypeFeature = features?.[ETrainsFeatureType.SUBTYPE];

        if (subTypeFeature?.type === ETrainsFeatureType.SUBTYPE) {
            const {title} = subTypeFeature;

            return (
                <Text size={size} color="highlight">
                    {CHAR_SPACE}
                    {CHAR_SPACE}
                    {title}
                </Text>
            );
        }

        return null;
    }, [features, size]);

    const namedTrainFeatureNode = useMemo<ReactNode>(() => {
        const trainName = getTrainNameByFeatures({
            features,
            checkSubTypeFeature: true,
        });

        if (trainName) {
            return (
                <Text size={size} color="highlight">
                    {CHAR_SPACE}
                    {CHAR_SPACE}
                    <span
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'firm',
                        })}
                    >
                        {trainName}
                    </span>
                </Text>
            );
        }

        return null;
    }, [features, size]);

    return (
        <div className={className}>
            {trainNumberNode}
            {subTypeFeatureNode}
            {namedTrainFeatureNode}
            {throughTrainFeatureNode}
            {companyNode}
        </div>
    );
};

export default TrainsSegmentInfo;
