import {useRef, useMemo, FC, ReactNode} from 'react';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {ITrainsMinTariffClass} from 'types/trains/common/tariffs/ITrainsMinTariffsClass';
import {ITrainsVariant} from 'types/trains/common/variant/ITrainsVariant';
import {ITrainsFilledSearchContext} from 'reducers/trains/context/types';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {deviceMods} from 'utilities/stylesUtils';
import useTooltipHover from 'utilities/hooks/useTooltipHover';
import getTariffClass from 'projects/trains/lib/getTariffClass';

import Flex from 'components/Flex/Flex';
import Price from 'components/Price/Price';

import ScopeContext from 'contexts/ScopeContext';

import TrainsSegmentTariffLink from '../TrainsSegmentTariffLink/TrainsSegmentTariffLink';
import TrainsSegmentTariffPlacesDetailsTooltip from '../TrainsSegmentTariffPlacesDetailsTooltip/TrainsSegmentTariffPlacesDetailsTooltip';

import cx from './TrainsSegmentTariff.scss';

export enum ETrainsSegmentTariffView {
    TEXT = 'text',
    LINK = 'link',
}

interface ITrainsSegmentTariffProps
    extends IWithClassName,
        IWithQaAttributes,
        IWithDeviceType {
    tariff?: ITrainsMinTariffClass;
    variant: ITrainsVariant;
    context: ITrainsFilledSearchContext;
    tariffViewType: ETrainsSegmentTariffView;
}

const TOOLTIP_HOVER_DELAY = 100;

const TrainsSegmentTariff: FC<ITrainsSegmentTariffProps> = props => {
    const {tariff, variant, context, className, deviceType, tariffViewType} =
        props;
    const {isDesktop} = deviceType;
    const isLinkView = tariffViewType === ETrainsSegmentTariffView.LINK;
    const triggerRef = useRef(null);
    const {isHovered, handleMouseEnter, handleMouseLeave} =
        useTooltipHover(TOOLTIP_HOVER_DELAY);

    const tagProps = useMemo(
        () => ({
            onMouseEnter: handleMouseEnter,
            onMouseLeave: handleMouseLeave,
        }),
        [handleMouseEnter, handleMouseLeave],
    );

    if (!tariff?.price || !tariff?.seats || !variant.orderUrl) {
        return null;
    }

    const {type, price, seats, placesDetails} = tariff;
    const tariffQaAttributes = prepareQaAttributes({
        parent: props,
        key: type,
    });

    const hasPlacesDetailsTooltip = placesDetails && isDesktop;

    const tariffTextNode = (
        <Flex
            className={cx(
                'tariff',
                {
                    tariff_hasPlacesDetailsTooltip: hasPlacesDetailsTooltip,
                },
                deviceMods('tariff', deviceType),
            )}
            nowrap
            inline
            flexDirection={deviceType.isDesktop ? 'row' : 'column'}
            alignItems={deviceType.isDesktop ? 'initial' : 'flex-start'}
            justifyContent={deviceType.isDesktop ? 'space-between' : 'initial'}
            between={deviceType.isDesktop ? 2 : 0}
            ref={triggerRef}
            tagProps={tagProps}
        >
            <Flex
                className={cx('tariffAndSeats')}
                justifyContent={
                    deviceType.isDesktop ? 'space-between' : 'initial'
                }
            >
                <span
                    className={cx('tariffName')}
                    {...prepareQaAttributes({
                        parent: tariffQaAttributes,
                        current: 'title',
                    })}
                >
                    {getTariffClass(type, deviceType.isDesktop)}
                </span>
                <span
                    {...prepareQaAttributes({
                        parent: tariffQaAttributes,
                        current: 'seats',
                    })}
                >
                    {seats}
                </span>
            </Flex>
            <Price
                className={cx('price')}
                isFrom
                isRound
                {...price}
                {...prepareQaAttributes({
                    parent: tariffQaAttributes,
                    current: 'price',
                })}
            />
        </Flex>
    );

    return (
        <ScopeContext.Consumer>
            {(scope): ReactNode => (
                <div
                    className={cx(className)}
                    key={type}
                    {...tariffQaAttributes}
                >
                    {isLinkView ? (
                        <TrainsSegmentTariffLink
                            variant={variant}
                            context={context}
                            tariff={tariff}
                        >
                            {tariffTextNode}
                        </TrainsSegmentTariffLink>
                    ) : (
                        tariffTextNode
                    )}
                    {hasPlacesDetailsTooltip && placesDetails && (
                        <TrainsSegmentTariffPlacesDetailsTooltip
                            isVisible={isHovered}
                            triggerRef={triggerRef}
                            placesDetails={placesDetails}
                            scope={scope}
                        />
                    )}
                </div>
            )}
        </ScopeContext.Consumer>
    );
};

export default TrainsSegmentTariff;
