import React, {ReactNode, useMemo, memo} from 'react';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {ITrain} from 'types/trains/common/train/ITrain';
import {ITrainsMinTariffs} from 'types/trains/common/tariffs/ITrainsMinTariffs';
import {ITrainsMinTariffClass} from 'types/trains/common/tariffs/ITrainsMinTariffsClass';
import {ITrainsVariant} from 'types/trains/common/variant/ITrainsVariant';
import {ITrainsFilledSearchContext} from 'reducers/trains/context/types';
import {TTrainsFeatures} from 'types/trains/common/features/TTrainsFeatures';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {deviceMods} from 'utilities/stylesUtils';
import {checkHasTrainsVariantTransfer} from 'projects/trains/lib/genericSearch/variants/checkHasTrainsVariantTransfer';
import {prepareAndSortTariffListByMinPrice} from 'projects/trains/lib/genericSearch/tariffs/prepareAndSortTariffListByMinPrice';
import {checkTariffsWithNotEmptyPlacesDetails} from 'projects/trains/components/TrainsSearchVariants/components/TrainsSearchSegment/components/TrainsSegmentTariffsPlacesDetails/utilities/filterTariffsWithEmptyPlaceDetailTypes';

import Flex from 'components/Flex/Flex';

import TrainsSegmentTariff, {
    ETrainsSegmentTariffView,
} from '../TrainsSegmentTariff/TrainsSegmentTariff';
import TrainsSegmentTariffsPlacesDetailsBottomSheet from '../TrainsSegmentTariffsPlacesDetailsBottomSheet/TrainsSegmentTariffsPlacesDetailsBottomSheet';

import cx from './TrainsSegmentTariffs.scss';

interface ITrainsSegmentTariffsProps
    extends IWithClassName,
        IWithQaAttributes,
        IWithDeviceType {
    variant: ITrainsVariant;
    context: ITrainsFilledSearchContext;
    train: ITrain;
    tariffViewType: ETrainsSegmentTariffView;
    tariffs?: ITrainsMinTariffs;
    features?: TTrainsFeatures;
    isPinnedSegment?: boolean;
}

const TrainsSegmentTariffs: React.FC<ITrainsSegmentTariffsProps> = props => {
    const {
        className,
        tariffs,
        variant,
        context,
        train,
        features,
        isPinnedSegment,
        tariffViewType,
        deviceType,
    } = props;
    const sortedTariffs = useMemo<ITrainsMinTariffClass[]>(() => {
        return prepareAndSortTariffListByMinPrice(tariffs);
    }, [tariffs]);

    const hasTariffsWithNotEmptyPlacesDetails = useMemo(() => {
        return checkTariffsWithNotEmptyPlacesDetails(sortedTariffs);
    }, [sortedTariffs]);

    const isVariantWithTransfer = useMemo(() => {
        return checkHasTrainsVariantTransfer({variant});
    }, [variant]);

    const tariffListNode = useMemo<ReactNode>(() => {
        if (!sortedTariffs.length) {
            return null;
        }

        return (
            <div
                className={cx(
                    'tariffList',
                    className,
                    deviceMods('tariffList', deviceType),
                )}
            >
                {sortedTariffs.map(
                    (tariff: ITrainsMinTariffClass): ReactNode => (
                        <TrainsSegmentTariff
                            key={tariff.type}
                            className={cx('tariff')}
                            deviceType={deviceType}
                            tariff={tariff}
                            variant={variant}
                            context={context}
                            tariffViewType={tariffViewType}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'tariff',
                            })}
                        />
                    ),
                )}
            </div>
        );
    }, [sortedTariffs, className, deviceType]);

    if (!tariffListNode) {
        return null;
    }

    if (deviceType.isDesktop) {
        return <div>{tariffListNode}</div>;
    }

    return (
        <Flex
            className={cx('tariffListContainer', {
                tariffListContainer_hasPlacesDetails:
                    hasTariffsWithNotEmptyPlacesDetails,
            })}
        >
            <div className={cx('scrollableWrapper')}>
                <div className={cx('scrollable')}>
                    <div className={cx('scrollableContent')}>
                        {tariffListNode}
                    </div>
                </div>
            </div>
            {hasTariffsWithNotEmptyPlacesDetails && (
                <TrainsSegmentTariffsPlacesDetailsBottomSheet
                    className={cx('placesDetailsBottomSheet')}
                    variant={variant}
                    context={context}
                    train={train}
                    features={features}
                    tariffs={sortedTariffs}
                    deviceType={deviceType}
                    hasVariantAction={
                        !isPinnedSegment && !isVariantWithTransfer
                    }
                    triggerClassName={cx('placesDetailsTriggerButton')}
                />
            )}
        </Flex>
    );
};

export default memo(TrainsSegmentTariffs);
