import React from 'react';
import moment from 'moment-timezone';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';

import {deviceMods} from 'utilities/stylesUtils';
import {CHAR_NBSP} from 'utilities/strings/charCodes';

import * as trainsSegmentKeyset from 'i18n/trains-segment';

import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import TrainsTime from 'projects/trains/components/TrainsTime/TrainsTime';
import TriggerDetailsButton from 'components/TriggerDetailsButton/TriggerDetailsButton';

import cx from './TrainsSegmentTimeRange.scss';

interface ITrainsSegmentTimeRangeProps extends IWithClassName, IWithDeviceType {
    minTime?: string;
    maxTime?: string;
    timezone: string;
    isDateVisible?: boolean;
}

const TrainsSegmentTimeRange: React.FC<ITrainsSegmentTimeRangeProps> = ({
    className,
    minTime,
    maxTime,
    timezone,
    deviceType,
    isDateVisible,
}) => {
    const descriptionNode = (
        <div
            className={cx('description', deviceMods('description', deviceType))}
        >
            <TriggerDetailsButton className={cx('descriptionIcon')}>
                <div className={cx(deviceMods('descriptionText', deviceType))}>
                    {trainsSegmentKeyset.arrivalDashRange()}
                </div>
            </TriggerDetailsButton>
        </div>
    );

    return (
        <Flex
            className={className}
            nowrap
            inline
            alignItems={deviceType.isDesktop ? 'center' : 'flex-end'}
        >
            <TrainsTime
                className={cx(
                    'time',
                    className,
                    deviceMods('time', deviceType),
                )}
                dateClassName={cx('date', deviceMods('date', deviceType))}
                timeMoment={moment.tz(minTime, timezone)}
                canShowDate={isDateVisible && deviceType.isMobile}
                deviceType={deviceType}
            />
            <Text size={deviceType.isDesktop ? 'xl' : 'l'}>
                {`...${CHAR_NBSP}`}
            </Text>
            <TrainsTime
                className={cx(
                    'time',
                    className,
                    deviceMods('time', deviceType),
                )}
                dateClassName={cx('date', deviceMods('date', deviceType))}
                timeMoment={moment.tz(maxTime, timezone)}
                canShowDate={isDateVisible}
                deviceType={deviceType}
                separator={deviceType.isDesktop ? descriptionNode : null}
            />

            {deviceType.isMobile && descriptionNode}
        </Flex>
    );
};

export default TrainsSegmentTimeRange;
