import React, {memo} from 'react';

import {EDirection} from 'types/common/EDirection';
import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {ITrainsFilledSearchContext} from 'reducers/trains/context/types';
import {ITrainsVariant} from 'types/trains/common/variant/ITrainsVariant';
import {EQueryingStatus} from 'types/trains/search/searchInfo/ITrainsSearchInfo';
import {ITrainsSearchBadgesInfo} from 'types/trains/search/badgesInfo/ITrainsSearchBadgesInfo';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {getTrainsVariantFirstSegment} from 'projects/trains/lib/genericSearch/variants/getTrainsVariantFirstSegment';

import TrainsSearchSegment from 'projects/trains/components/TrainsSearchVariants/components/TrainsSearchSegment/TrainsSearchSegment';
import CardWithDeviceLayout from 'components/CardWithDeviceLayout/CardWithDeviceLayout';

import {useTrainsSearchVariantBadges} from 'projects/trains/pages/TrainsSearchPage/hooks/useTrainsSearchVariantBadges';

import TrainsSearchVariantBadges from '../TrainsSearchVariantBadges/TrainsSearchVariantBadges';

import cx from './TrainsSearchVariant.scss';

interface ITrainsSearchVariantProps
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    variant: ITrainsVariant;
    context: ITrainsFilledSearchContext;
    searchStatus: EQueryingStatus;
    badgesInfo: ITrainsSearchBadgesInfo;
    direction: EDirection;
}

const TrainsSearchVariant: React.FC<ITrainsSearchVariantProps> = props => {
    const {variant, context, searchStatus, deviceType, badgesInfo, direction} =
        props;
    const variantBadges = useTrainsSearchVariantBadges({
        variant,
        badgesInfo,
        deviceType,
    });
    const segment = getTrainsVariantFirstSegment({variant, direction});

    if (segment) {
        const segmentNode = (
            <TrainsSearchSegment
                segment={segment}
                variant={variant}
                context={context}
                direction={direction}
                searchStatus={searchStatus}
                variantBadges={variantBadges}
                {...prepareQaAttributes({
                    key: 0,
                    parent: props,
                    current: 'segment',
                })}
            />
        );

        if (deviceType.isDesktop) {
            return (
                <CardWithDeviceLayout className={cx('root')}>
                    {segmentNode}
                    <TrainsSearchVariantBadges variantBadges={variantBadges} />
                </CardWithDeviceLayout>
            );
        }

        return segmentNode;
    }

    return null;
};

export default memo(TrainsSearchVariant);
