import React, {useCallback} from 'react';

import {ITrainsFilledSearchContext} from 'reducers/trains/context/types';
import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {ETrainsGoal} from 'utilities/metrika/types/goals/trains';
import {ETrainsVariantUrlOwner} from 'types/trains/common/variant/ETrainsVariantUrlOwner';
import {ITrainsVariant} from 'types/trains/common/variant/ITrainsVariant';

import {reachGoal} from 'utilities/metrika';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {getTrainsVariantActionUrl} from 'projects/trains/lib/genericSearch/variants/getTrainsVariantActionUrl';
import {getTrainsVariantActionText} from 'projects/trains/lib/genericSearch/variants/getTrainsVariantActionText';
import {checkHasTrainsVariantTransfer} from 'projects/trains/lib/genericSearch/variants/checkHasTrainsVariantTransfer';

import Text from 'components/Text/Text';
import Link from 'components/Link/Link';
import ButtonLink from 'components/ButtonLink/ButtonLink';

import cx from './TrainsSearchVariantAction.scss';

interface ITrainsSearchVariantActionProps
    extends IWithQaAttributes,
        IWithClassName,
        IWithDeviceType {
    context: ITrainsFilledSearchContext;
    variant: ITrainsVariant;
    isPinnedSegment?: boolean;
    onActionClick?: () => void;
}

const TrainsSearchVariantAction: React.FC<ITrainsSearchVariantActionProps> =
    props => {
        const {
            className,
            variant,
            context,
            deviceType,
            onActionClick,
            isPinnedSegment,
        } = props;
        const {orderUrl} = variant;
        const {isMobile} = deviceType;
        const handleExternalClick = useCallback(() => {
            reachGoal(ETrainsGoal.ORDER_UFS_LINK_CLICK);
        }, []);

        if (!orderUrl) {
            return null;
        }

        const isVariantWithTransfer = checkHasTrainsVariantTransfer({variant});
        const variantActionText = getTrainsVariantActionText({
            context,
            isPinnedSegment,
            owner: orderUrl.owner,
            isVariantWithTransfer,
        });
        const variantActionUrl = getTrainsVariantActionUrl({
            variant,
            context,
            orderUrl,
            isPinnedSegment,
        });

        if (orderUrl.owner === ETrainsVariantUrlOwner.UFS) {
            return (
                <ButtonLink
                    className={className}
                    theme="primary"
                    size={isMobile ? 'l' : 'm-inset'}
                    width="max"
                    url={variantActionUrl}
                    onClick={handleExternalClick}
                    target="_blank"
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'ufsActionButton',
                    })}
                >
                    {variantActionText}
                </ButtonLink>
            );
        }

        if (isPinnedSegment) {
            return (
                <Text
                    className={cx('pinnedSegmentLink', className)}
                    tag="div"
                    size={isMobile ? 's' : 'm'}
                    align={isMobile ? 'left' : 'center'}
                >
                    <Link
                        to={variantActionUrl}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'changeSegmentLink',
                        })}
                    >
                        {variantActionText}
                    </Link>
                </Text>
            );
        }

        return (
            <ButtonLink
                className={className}
                theme="primary"
                size={isMobile ? 'l' : 'm-inset'}
                width="max"
                to={variantActionUrl}
                onClick={onActionClick}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'boyActionButton',
                })}
            >
                {variantActionText}
            </ButtonLink>
        );
    };

export default TrainsSearchVariantAction;
