import React, {ReactNode, useMemo} from 'react';

import {EDirection} from 'types/common/EDirection';
import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {ITrainsFilledSearchContext} from 'reducers/trains/context/types';
import {EQueryingStatus} from 'types/trains/search/searchInfo/ITrainsSearchInfo';
import {ITrainsVariant} from 'types/trains/common/variant/ITrainsVariant';
import {ETrainsBrokenClassesCode} from 'types/trains/common/tariffs/ETrainsBrokenClassesCode';

import IPrice from 'utilities/currency/PriceInterface';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {getTrainsVariantMinPrice} from 'projects/trains/lib/genericSearch/variants/getTrainsVariantMinPrice';
import {getTrainsVariantSegments} from 'projects/trains/lib/genericSearch/variants/getTrainsVariantSegments';
import {getSegmentMainTariffTitleByBrokenClassesCode} from 'projects/trains/lib/genericSearch/tariffs/getSegmentMainTariffTitleByBrokenClassesCode';
import {getVariantBrokenClassesCode} from 'projects/trains/components/TrainsSearchVariants/components/TrainsSearchVariantPriceAndAction/utilities/getVariantBrokenClassesCode';

import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import Price from 'components/Price/Price';
import Spinner from 'components/Spinner/Spinner';

import TrainsSearchVariantAction from '../TrainsSearchVariantAction/TrainsSearchVariantAction';

import cx from './TrainsSearchVariantPriceAndAction.scss';

interface ITrainsSearchVariantPriceAndActionProps
    extends IWithClassName,
        IWithQaAttributes,
        IWithDeviceType {
    variant: ITrainsVariant;
    context: ITrainsFilledSearchContext;
    direction: EDirection;
    searchStatus: EQueryingStatus;
    canHidePrice?: boolean;
    isPinnedSegment?: boolean;
}

const TrainsSearchVariantPriceAndAction: React.FC<ITrainsSearchVariantPriceAndActionProps> =
    props => {
        const {
            context,
            variant,
            className,
            direction,
            deviceType,
            canHidePrice,
            searchStatus,
            isPinnedSegment,
        } = props;

        const variantMinPrice = useMemo<IPrice | undefined>(() => {
            return getTrainsVariantMinPrice({variant, direction});
        }, [variant, direction]);

        const variantSegments = getTrainsVariantSegments({variant, direction});
        const variantBrokenClassesCode = useMemo<
            ETrainsBrokenClassesCode | undefined
        >(() => {
            return getVariantBrokenClassesCode(variantSegments);
        }, [variantSegments]);

        const contentNode = ((): ReactNode => {
            if (!variantMinPrice) {
                const variantTitleByBrokenClassesCode =
                    getSegmentMainTariffTitleByBrokenClassesCode(
                        variantBrokenClassesCode,
                    );

                if (searchStatus === EQueryingStatus.QUERYING) {
                    return (
                        <span
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'pricesSpinner',
                            })}
                        >
                            <Spinner size="s" />
                        </span>
                    );
                }

                if (variantTitleByBrokenClassesCode) {
                    return (
                        <Text size="m">{variantTitleByBrokenClassesCode}</Text>
                    );
                }

                return null;
            }

            if (!variant.orderUrl) {
                return null;
            }

            if (variantMinPrice) {
                return (
                    <>
                        {!canHidePrice && (
                            <Price
                                className={cx('price')}
                                isFrom
                                isRound
                                value={variantMinPrice.value}
                                currency={variantMinPrice.currency}
                                {...prepareQaAttributes({
                                    parent: props,
                                    current: 'variantMinPrice',
                                })}
                            />
                        )}

                        <TrainsSearchVariantAction
                            variant={variant}
                            context={context}
                            deviceType={deviceType}
                            isPinnedSegment={isPinnedSegment}
                            {...prepareQaAttributes(props)}
                        />
                    </>
                );
            }

            return null;
        })();

        if (!contentNode) {
            return null;
        }

        return (
            <Flex
                className={cx('root', className)}
                between="2"
                flexDirection="column"
                alignItems="center"
                justifyContent="center"
            >
                {contentNode}
            </Flex>
        );
    };

export default TrainsSearchVariantPriceAndAction;
