import React, {useCallback} from 'react';

import {EDirection} from 'types/common/EDirection';
import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {ETrainsGoal} from 'utilities/metrika/types/goals/trains';
import {ITrainsVariant} from 'types/trains/common/variant/ITrainsVariant';
import {ITrainsFilledSearchContext} from 'reducers/trains/context/types';
import {EQueryingStatus} from 'types/trains/search/searchInfo/ITrainsSearchInfo';
import {ISearchSegmentBadge} from 'components/SearchSegmentBadges/types/ISearchSegmentBadge';

import {reachGoal} from 'utilities/metrika';
import {useToggle} from 'utilities/hooks/useToggle';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {getTrainsTotalDurationBySegments} from 'projects/trains/lib/genericSearch/transfers/getTrainsTotalDurationBySegments';

import * as i18nBlock from 'i18n/trains-common';

import Flex from 'components/Flex/Flex';
import Card from 'components/Card/Card';
import Box from 'components/Box/Box';
import LinkButton from 'components/LinkButton/LinkButton';
import RotatingArrowIcon from 'components/RotatingArrowIcon/RotatingArrowIcon';
import TrainsSegmentPoints from 'projects/trains/components/TrainsSearchVariants/components/TrainsSearchSegment/components/TrainsSegmentPoints/TrainsSegmentPoints';
import TrainsSegmentDuration from 'projects/trains/components/TrainsSearchVariants/components/TrainsSearchSegment/components/TrainsSegmentDuration/TrainsSegmentDuration';
import TrainsSearchVariantTransferSegments from '../components/TrainsSearchVariantTransferSegments/TrainsSearchVariantTransferSegments';
import TrainsSearchVariantTransfers from '../components/TrainsSearchVariantTransfers/TrainsSearchVariantTransfers';

import TrainsSearchVariantBadges from '../../TrainsSearchVariantBadges/TrainsSearchVariantBadges';
import TrainsSearchVariantPriceAndAction from '../../TrainsSearchVariantPriceAndAction/TrainsSearchVariantPriceAndAction';

import cx from './TrainsDesktopSearchVariantWithTransfer.scss';

export interface ITrainsDesktopSearchVariantWithTransferProps
    extends IWithQaAttributes,
        IWithClassName,
        IWithDeviceType {
    variant: ITrainsVariant;
    context: ITrainsFilledSearchContext;
    direction: EDirection;
    searchStatus: EQueryingStatus;
    variantBadges: ISearchSegmentBadge[];
}

const TrainsDesktopSearchVariantWithTransfer: React.FC<ITrainsDesktopSearchVariantWithTransferProps> =
    props => {
        const {
            variant,
            context,
            searchStatus,
            deviceType,
            variantBadges,
            direction,
        } = props;
        const {forward} = variant;
        const [firstSegment] = forward;
        const lastSegment = forward[forward.length - 1];
        const transferTotalDuration = getTrainsTotalDurationBySegments({
            firstSegment,
            lastSegment,
        });
        const [isVisibleTransferSegments, toggleTransferSegmentsVisibility] =
            useToggle(false);

        const handleClickTransferSegmentsVisibility = useCallback(() => {
            if (!isVisibleTransferSegments) {
                reachGoal(ETrainsGoal.TRANSFER_SHOW_ROUTE);
            }

            toggleTransferSegmentsVisibility();
        }, [toggleTransferSegmentsVisibility, isVisibleTransferSegments]);

        const componentQaAttributes: IWithQaAttributes =
            prepareQaAttributes(props);

        return (
            <Card
                shadow="default"
                className={cx('root', {
                    root_isVisibleTransferSegments: isVisibleTransferSegments,
                })}
            >
                <TrainsSearchVariantBadges variantBadges={variantBadges} />
                <Flex
                    className={cx('transferSummary')}
                    x={8}
                    y={7}
                    justifyContent="space-between"
                >
                    <Flex nowrap inline between={4}>
                        <TrainsSegmentPoints
                            stationFrom={firstSegment.stationFrom}
                            stationTo={lastSegment.stationTo}
                            departure={firstSegment.departure}
                            arrival={lastSegment.arrival}
                            deviceType={deviceType}
                            {...componentQaAttributes}
                        />
                        <Box between="1">
                            <TrainsSegmentDuration
                                duration={transferTotalDuration}
                                deviceType={deviceType}
                                {...prepareQaAttributes({
                                    parent: componentQaAttributes,
                                    current: 'duration',
                                })}
                            />
                            <TrainsSearchVariantTransfers
                                variant={variant}
                                deviceType={deviceType}
                                {...componentQaAttributes}
                            />
                            <LinkButton
                                className={cx('toggleLink')}
                                onClick={handleClickTransferSegmentsVisibility}
                                {...prepareQaAttributes({
                                    parent: componentQaAttributes,
                                    current: 'toggleSegmentsVisibilityLink',
                                })}
                            >
                                {i18nBlock.routeDashAndDashPrices()}
                                <RotatingArrowIcon
                                    className={cx('toggleArrow')}
                                    rotated={isVisibleTransferSegments}
                                />
                            </LinkButton>
                        </Box>
                    </Flex>
                    <div className={cx('priceAndAction')}>
                        <TrainsSearchVariantPriceAndAction
                            variant={variant}
                            context={context}
                            direction={direction}
                            searchStatus={searchStatus}
                            deviceType={deviceType}
                        />
                    </div>
                </Flex>
                {isVisibleTransferSegments && (
                    <Box x={8} y={7}>
                        <TrainsSearchVariantTransferSegments
                            variant={variant}
                            context={context}
                            direction={direction}
                            searchStatus={searchStatus}
                            deviceType={deviceType}
                            {...componentQaAttributes}
                        />
                    </Box>
                )}
            </Card>
        );
    };

export default TrainsDesktopSearchVariantWithTransfer;
