import React, {useMemo, useCallback} from 'react';

import {EDirection} from 'types/common/EDirection';
import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {ETrainsGoal} from 'utilities/metrika/types/goals/trains';
import {ITrainsVariant} from 'types/trains/common/variant/ITrainsVariant';
import {ITrainsFilledSearchContext} from 'reducers/trains/context/types';
import {EQueryingStatus} from 'types/trains/search/searchInfo/ITrainsSearchInfo';
import {ISearchSegmentBadge} from 'components/SearchSegmentBadges/types/ISearchSegmentBadge';

import IPrice from 'utilities/currency/PriceInterface';
import {useToggle} from 'utilities/hooks/useToggle';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {reachGoal} from 'utilities/metrika';
import {getTrainsVariantMinPrice} from 'projects/trains/lib/genericSearch/variants/getTrainsVariantMinPrice';
import {getTrainsTotalDurationBySegments} from 'projects/trains/lib/genericSearch/transfers/getTrainsTotalDurationBySegments';

import * as trainsCommonKeyset from 'i18n/trains-common';

import Box from 'components/Box/Box';
import Flex from 'components/Flex/Flex';
import Card from 'components/Card/Card';
import Price from 'components/Price/Price';
import ModalWithBackButton from 'components/ModalWithBackButton/ModalWithBackButton';
import TrainsSearchVariantTransferSegments from '../components/TrainsSearchVariantTransferSegments/TrainsSearchVariantTransferSegments';
import TrainsSegmentTimings from 'projects/trains/components/TrainsSearchVariants/components/TrainsSearchSegment/components/TrainsSegmentTimings/TrainsSegmentTimings';
import TrainsSegmentStations from 'projects/trains/components/TrainsSearchVariants/components/TrainsSearchSegment/components/TrainsSegmentStations/TrainsSegmentStations';
import TrainsSearchVariantTransfers from '../components/TrainsSearchVariantTransfers/TrainsSearchVariantTransfers';
import LinkButton from 'components/LinkButton/LinkButton';

import TrainsSearchVariantBadges from '../../TrainsSearchVariantBadges/TrainsSearchVariantBadges';
import TrainsSearchVariantPriceAndAction from '../../TrainsSearchVariantPriceAndAction/TrainsSearchVariantPriceAndAction';

import cx from './TrainsMobileSearchVariantWithTransfer.scss';

export interface ITrainsMobileSearchVariantWithTransferProps
    extends IWithQaAttributes,
        IWithClassName,
        IWithDeviceType {
    variant: ITrainsVariant;
    context: ITrainsFilledSearchContext;
    direction: EDirection;
    searchStatus: EQueryingStatus;
    variantBadges: ISearchSegmentBadge[];
}

const TrainsMobileSearchVariantWithTransfer: React.FC<ITrainsMobileSearchVariantWithTransferProps> =
    props => {
        const {
            variant,
            context,
            searchStatus,
            deviceType,
            variantBadges,
            direction,
        } = props;
        const {forward} = variant;
        const [firstSegment] = forward;
        const lastSegment = forward[forward.length - 1];
        const transferTotalDuration = getTrainsTotalDurationBySegments({
            firstSegment,
            lastSegment,
        });
        const [isVisibleTransferSegments, toggleTransferSegmentsVisibility] =
            useToggle(false);
        const handleClickTransferSegmentsVisibility = useCallback(() => {
            toggleTransferSegmentsVisibility();
            reachGoal(ETrainsGoal.TRANSFER_SHOW_ROUTE);
        }, [toggleTransferSegmentsVisibility]);

        const variantMinPrice = useMemo<IPrice | undefined>(() => {
            return getTrainsVariantMinPrice({variant});
        }, [variant]);

        const componentQaAttributes: IWithQaAttributes =
            prepareQaAttributes(props);

        return (
            <Card
                shadow="default"
                className={cx('root', {
                    root_hasBadges: Boolean(variantBadges.length),
                })}
            >
                <TrainsSearchVariantBadges variantBadges={variantBadges} />
                <Flex flexDirection="column">
                    <TrainsSegmentTimings
                        className={cx('timings')}
                        stationFrom={firstSegment.stationFrom}
                        stationTo={lastSegment.stationTo}
                        departure={firstSegment.departure}
                        arrival={lastSegment.arrival}
                        duration={transferTotalDuration}
                        deviceType={deviceType}
                        {...componentQaAttributes}
                    />
                    <TrainsSegmentStations
                        className={cx('stations')}
                        stationFrom={firstSegment.stationFrom}
                        stationTo={lastSegment.stationTo}
                        {...componentQaAttributes}
                    />
                    <TrainsSearchVariantTransfers
                        className={cx('transfers')}
                        variant={variant}
                        deviceType={deviceType}
                        {...componentQaAttributes}
                    />
                    <Box above="1">
                        <LinkButton
                            className={cx('linkButton')}
                            onClick={handleClickTransferSegmentsVisibility}
                            {...prepareQaAttributes({
                                parent: componentQaAttributes,
                                current: 'toggleSegmentsVisibilityLink',
                            })}
                        >
                            {trainsCommonKeyset.routeDashAndDashPrices()}
                        </LinkButton>
                    </Box>
                    <div
                        className={cx('footer', {
                            footer_hasTariffs: Boolean(variantMinPrice),
                        })}
                    >
                        {variantMinPrice ? (
                            <Price
                                isFrom
                                isRound
                                value={variantMinPrice.value}
                                currency={variantMinPrice.currency}
                            />
                        ) : null}
                        <TrainsSearchVariantPriceAndAction
                            className={cx('action')}
                            context={context}
                            variant={variant}
                            direction={direction}
                            searchStatus={searchStatus}
                            deviceType={deviceType}
                            canHidePrice
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'cardVariantAction',
                            })}
                        />
                    </div>
                </Flex>
                <ModalWithBackButton
                    backButtonText={trainsCommonKeyset.backDashToDashSearchDashResults()}
                    canRenderBackButton
                    verticalAlign="top"
                    fullScreen
                    onClose={toggleTransferSegmentsVisibility}
                    isVisible={isVisibleTransferSegments}
                    {...componentQaAttributes}
                >
                    <div className={cx('segmentsWithVariantAction')}>
                        <TrainsSearchVariantTransferSegments
                            variant={variant}
                            context={context}
                            direction={direction}
                            searchStatus={searchStatus}
                            deviceType={deviceType}
                            {...componentQaAttributes}
                        />
                        <TrainsSearchVariantPriceAndAction
                            className={cx('modalVariantAction')}
                            context={context}
                            variant={variant}
                            direction={direction}
                            searchStatus={searchStatus}
                            deviceType={deviceType}
                            canHidePrice
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'modalVariantAction',
                            })}
                        />
                    </div>
                </ModalWithBackButton>
            </Card>
        );
    };

export default TrainsMobileSearchVariantWithTransfer;
