import moment from 'moment';

import {SUBSCRIPTION_BLOCK_OFFSET} from 'constants/subscription';

import {
    TTrainsVariantId,
    TTrainsVariantWithVisibleStatusById,
} from 'types/trains/common/variant/ITrainsVariant';
import {ITrainsFilledSearchContext} from 'reducers/trains/context/types';
import {
    ESearchListItemType,
    TSearchListItem,
} from '../types/ITrainsSearchVariants';
import {ITrainsSearchSort} from 'projects/trains/lib/sort/types';
import {ESortType} from 'types/common/sort/ESortType';

import {atPlaceOrEarly} from 'utilities/actions/atPlaceOrEarly';
import getNextDaySeparatorInfo from 'projects/trains/components/TrainsSearchVariants/utilities/getNextDaySeparatorInfo';
import getNextSearchDayMoment from 'projects/trains/lib/context/getNextSearchDayMoment';
import {formatDate} from 'utilities/dateUtils';
import {HUMAN} from 'utilities/dateUtils/formats';

export const prepareTrainsSearchList = ({
    sort,
    context,
    variantIds,
    bannerIndexPosition,
    variantWithVisibleStatusById,
    trainsCrossSaleEnabled,
    trainsCleanSerpEnabled,
}: {
    sort: ITrainsSearchSort;
    context: ITrainsFilledSearchContext;
    variantIds: TTrainsVariantId[];
    bannerIndexPosition: number;
    variantWithVisibleStatusById: TTrainsVariantWithVisibleStatusById;
    trainsCrossSaleEnabled: boolean;
    trainsCleanSerpEnabled: boolean;
}): TSearchListItem[] => {
    const {direction} = context;
    let foundVisibleVariants = 0;
    let hasNextDaySeparatorAdded = false;
    let needToCheckNextDaySeparator = true;
    const separatorMoment = getNextSearchDayMoment(context);
    const isDepartureSorting = sort.by === ESortType.DEPARTURE;

    const bannerIndexPositionByCurrentList = Math.min(
        variantIds.length,
        bannerIndexPosition,
    );

    const visibleCount = variantIds.filter(
        variantId => variantWithVisibleStatusById[variantId].isVisible,
    ).length;

    return variantIds.reduce<TSearchListItem[]>((searchList, variantId) => {
        const {variant, isVisible} = variantWithVisibleStatusById[variantId];

        if (
            !hasNextDaySeparatorAdded &&
            needToCheckNextDaySeparator &&
            isVisible &&
            isDepartureSorting
        ) {
            const separatorInfo = getNextDaySeparatorInfo({
                variantAndDirection: {variant, direction},
                separatorMoment,
                isReverseSorting: sort.isReverse,
                isFirstVisibleSegment: foundVisibleVariants === 0,
            });

            if (separatorInfo.needToAdd) {
                hasNextDaySeparatorAdded = true;

                searchList.push({
                    type: ESearchListItemType.DAY_SEPARATOR,
                    date: formatDate(
                        moment(separatorMoment).subtract(
                            sort.isReverse ? 1 : 0,
                            'days',
                        ),
                        HUMAN,
                    ),
                });
            }

            needToCheckNextDaySeparator = separatorInfo.needToCheckFurther;
        }

        if (isVisible) {
            foundVisibleVariants++;
        }

        searchList.push({type: ESearchListItemType.VARIANT, variantId});

        if (
            !trainsCleanSerpEnabled &&
            foundVisibleVariants === bannerIndexPositionByCurrentList
        ) {
            searchList.push({type: ESearchListItemType.BANNER});
        }

        if (
            trainsCrossSaleEnabled &&
            atPlaceOrEarly(
                visibleCount,
                foundVisibleVariants - 1,
                bannerIndexPositionByCurrentList - 1,
            )
        ) {
            searchList.push({type: ESearchListItemType.HOTELS_CROSS_SALE});
        }

        if (
            !trainsCleanSerpEnabled &&
            atPlaceOrEarly(
                visibleCount,
                foundVisibleVariants - 1,
                bannerIndexPositionByCurrentList -
                    1 +
                    SUBSCRIPTION_BLOCK_OFFSET,
            )
        ) {
            searchList.push({type: ESearchListItemType.SUBSCRIPTION_FORM});
        }

        return searchList;
    }, []);
};
