import React from 'react';
import {Moment} from 'moment';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';

import {deviceMods} from 'utilities/stylesUtils';
import {TIME, HUMAN_SHORT} from 'utilities/dateUtils/formats';
import {
    prepareQaAttributes,
    IWithQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/trains-segment';

import cx from './TrainsTime.scss';

interface ITrainsTimeProps
    extends IWithClassName,
        IWithQaAttributes,
        IWithDeviceType {
    timeMoment: Moment;
    dateClassName?: string;
    timeClassName?: string;
    canShowDate?: boolean | null;
    dateFormat?: string;
    isNextDay?: boolean;
    separator?: React.ReactNode;
}

const TrainsTime: React.FC<ITrainsTimeProps> = props => {
    const {
        className,
        canShowDate,
        timeMoment,
        dateFormat = HUMAN_SHORT,
        dateClassName,
        timeClassName,
        isNextDay,
        deviceType,
        separator = null,
    } = props;

    return (
        <div
            className={cx(
                'dateTime',
                deviceMods('dateTime', deviceType),
                className,
            )}
            {...prepareQaAttributes(props)}
        >
            {canShowDate && (
                <div
                    className={cx('date', dateClassName)}
                    {...prepareQaAttributes({parent: props, current: 'date'})}
                >
                    {isNextDay
                        ? i18nBlock.nextDashDay({
                              day: timeMoment.format(dateFormat),
                          })
                        : timeMoment.format(dateFormat)}
                </div>
            )}

            {separator}

            <span
                className={cx('time', timeClassName)}
                {...prepareQaAttributes({parent: props, current: 'time'})}
            >
                {timeMoment.format(TIME)}
            </span>
        </div>
    );
};

export default TrainsTime;
