import React, {useMemo} from 'react';

import {IWithDeviceType} from 'types/withDeviceType';
import {IWithClassName} from 'types/withClassName';
import {ITrainsStation} from 'types/trains/common/station/ITrainsStation';

import {getTransferPointDescription} from 'projects/trains/lib/segments/transfers/transferPoint';
import getTrainsTransferHumanDuration from 'projects/trains/components/Segment/utilities/getTrainsTransferHumanDuration';
import {deviceMods} from 'utilities/stylesUtils';
import {CHAR_EM_DASH, CHAR_NBSP} from 'utilities/strings/charCodes';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import Separator from 'components/Separator/Separator';
import ConnectionIcon from 'icons/12/Connection';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import TextSkeleton from 'components/Skeletons/TextSkeleton/TextSkeleton';

import cx from './TransferSeparator.scss';

interface ISegment {
    stationFrom: ITrainsStation;
    stationTo: ITrainsStation;
    departure: string;
    arrival: string;
}

interface ITransferSeparatorProps
    extends IWithDeviceType,
        IWithClassName,
        IWithQaAttributes {
    currentSegment: ISegment;
    nextSegment: ISegment;
    hideSeparator?: boolean;
    loading?: boolean;
}

const TransferSeparator: React.FC<ITransferSeparatorProps> = props => {
    const {
        currentSegment,
        nextSegment,
        deviceType,
        className,
        hideSeparator = false,
        loading = false,
        ...rest
    } = props;

    const hasStationChange =
        currentSegment.stationTo.id !== nextSegment.stationFrom.id;

    const text = useMemo(() => {
        return (
            <span>
                <span
                    className={cx('point')}
                    {...prepareQaAttributes({
                        parent: rest,
                        current: 'description',
                    })}
                >
                    {getTransferPointDescription(currentSegment, nextSegment)}
                </span>
                {`${CHAR_NBSP}${CHAR_EM_DASH}${CHAR_NBSP}`}
                <span
                    className={cx('duration')}
                    {...prepareQaAttributes({
                        parent: rest,
                        current: 'duration',
                    })}
                >
                    {getTrainsTransferHumanDuration(
                        currentSegment,
                        nextSegment,
                    )}
                </span>
            </span>
        );
    }, [currentSegment, nextSegment]);

    return (
        <div
            className={cx('root', deviceMods('root', deviceType), className)}
            {...prepareQaAttributes(props)}
        >
            {loading ? (
                <TextSkeleton
                    size="m"
                    className={cx('skeleton')}
                    withAnimation
                />
            ) : (
                <>
                    <div className={cx('content')}>
                        {hasStationChange ? (
                            <TextWithIcon
                                size="m"
                                text={text}
                                iconLeft={ConnectionIcon}
                                iconLeftClassName={cx('changeIcon')}
                            />
                        ) : (
                            text
                        )}
                    </div>

                    {deviceType.isDesktop && !hideSeparator && (
                        <Separator className={cx('separator')} />
                    )}
                </>
            )}
        </div>
    );
};

export default TransferSeparator;
