import React, {Component} from 'react';

import {IWithClassName} from 'types/withClassName';
import {EBadgeTheme} from 'projects/trains/components/badges/Badge/types';
import EPopupDirection from 'components/Popup/types/EPopupDirection';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import MessageBoxPopup from 'components/MessageBoxPopup/MessageBoxPopup';

import DeviceTypeContext from 'contexts/DeviceTypeContext';

import cx from './Badge.scss';

const DELAY = 100;

const DIRECTIONS = [
    EPopupDirection.BOTTOM,
    EPopupDirection.TOP,
    EPopupDirection.BOTTOM_LEFT,
    EPopupDirection.TOP_LEFT,
];

interface IBadgeProps extends IWithClassName, IWithQaAttributes {
    badgeClassName?: string;
    theme?: EBadgeTheme;
    hint?: React.ReactNode;
}

interface IBadgeState {
    isVisible: boolean;
}

export default class Badge extends Component<IBadgeProps, IBadgeState> {
    state = {
        isVisible: false,
    };

    private rootRef = React.createRef<HTMLSpanElement>();

    private abbrRef = React.createRef<HTMLElement>();

    private tooltipTimeout: number | null = null;

    private onMouseEnter = (): void => {
        if (!this.props.hint) {
            return;
        }

        if (this.tooltipTimeout) {
            window.clearTimeout(this.tooltipTimeout);
        }

        this.tooltipTimeout = window.setTimeout(() => {
            this.setState({isVisible: true});

            this.tooltipTimeout = null;
        }, DELAY);
    };

    private onMouseLeave = (): void => {
        if (!this.props.hint) {
            return;
        }

        if (this.tooltipTimeout) {
            clearTimeout(this.tooltipTimeout);

            this.setState({isVisible: false});
        } else {
            this.tooltipTimeout = window.setTimeout(() => {
                this.setState({isVisible: false});

                this.tooltipTimeout = null;
            }, DELAY);
        }
    };

    render(): React.ReactNode {
        const {isVisible} = this.state;
        const {theme, hint, children, className, badgeClassName} = this.props;

        return (
            <DeviceTypeContext.Consumer>
                {(deviceType): React.ReactNode => (
                    <span
                        ref={this.rootRef}
                        className={cx('root', className)}
                        {...prepareQaAttributes(this.props)}
                        onMouseEnter={
                            deviceType.isDesktop ? this.onMouseEnter : undefined
                        }
                        onMouseLeave={
                            deviceType.isDesktop ? this.onMouseLeave : undefined
                        }
                    >
                        <abbr
                            ref={this.abbrRef}
                            className={cx(
                                'badge',
                                {
                                    badge_tooltipVisible: isVisible,
                                },
                                `badge_theme_${theme}`,
                                badgeClassName,
                            )}
                            aria-label={typeof hint === 'string' ? hint : ''}
                        >
                            {children}
                        </abbr>

                        {deviceType.isDesktop && hint && (
                            <MessageBoxPopup
                                isVisible={isVisible}
                                direction={DIRECTIONS}
                                anchorRef={this.abbrRef}
                                scopeRef={this.rootRef}
                            >
                                <div className={cx('hintText')}>{hint}</div>
                            </MessageBoxPopup>
                        )}
                    </span>
                )}
            </DeviceTypeContext.Consumer>
        );
    }
}
