import React, {ReactNode, useMemo} from 'react';

import {EProjectName} from 'constants/common';

import {EFooterProject} from 'components/Footer/types';
import EHeaderBorderBottomType from 'components/Header/types/EHeaderBorderBottomType';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import Container from 'components/Layouts/Container/Container';
import SearchForm from 'projects/trains/components/SearchForm/SearchForm';
import LayoutDefault from 'components/Layouts/LayoutDefault/LayoutDefault';
import SearchHeaderFormDesktopContainer from 'components/SearchHeaderFormDesktopContainer/SearchHeaderFormDesktopContainer';

import cx from './SearchLayout.scss';

interface ISearchLayoutProps {
    children: {
        content: ReactNode;
        filters: ReactNode;
        footer: ReactNode;
        fastFilters?: ReactNode;
        error?: ReactNode;
        breadcrumbs?: ReactNode;
    };
    isLoading: boolean;
}

const SearchLayout: React.FC<ISearchLayoutProps> = ({children, isLoading}) => {
    const deviceType = useDeviceType();
    const desktopFiltersAndSearchFormBarNode = useMemo(() => {
        if (deviceType.isMobile) {
            return null;
        }

        return (
            <SearchHeaderFormDesktopContainer
                searchFormNode={<SearchForm />}
                belowFormNode={children.filters}
            />
        );
    }, [deviceType, children.filters]);

    return (
        <LayoutDefault
            isFixedNavigation
            showNavigation
            showSearchForm={deviceType.isMobile}
            hasSideSheetNavigation
            headerBorderBottomType={
                deviceType.isDesktop
                    ? EHeaderBorderBottomType.NONE
                    : EHeaderBorderBottomType.FULL
            }
            filters={deviceType.isMobile ? children.filters : undefined}
            fastFilters={deviceType.isMobile ? children.fastFilters : undefined}
            searchFormInitialIsExpanded={false}
            project={EProjectName.TRAINS}
            footerType={EFooterProject.TRAINS}
            showFooter={!isLoading}
        >
            {children.error || (
                <>
                    {desktopFiltersAndSearchFormBarNode}
                    {children.breadcrumbs}
                    {children.content && (
                        <Container
                            className={cx(
                                'content',
                                deviceMods('content', deviceType),
                                {
                                    content_loading: isLoading,
                                },
                            )}
                            provideScope
                        >
                            {children.content}
                        </Container>
                    )}
                    {children.footer && (
                        <Container
                            className={cx(deviceMods('footer', deviceType), {
                                footer_loading: isLoading,
                            })}
                        >
                            {children.footer}
                        </Container>
                    )}
                </>
            )}
        </LayoutDefault>
    );
};

export default SearchLayout;
