import React, {useContext} from 'react';
import {useSelector} from 'react-redux';
import Helmet from 'react-helmet';
import {stringify as qsStringify} from 'query-string';

import {
    isFilledTrainsSearchContext,
    TrainsSearchContextType,
} from 'reducers/trains/context/types';

import {trainsContextSelector} from 'selectors/trains/trainsContextSelector';
import nonceSelector from 'selectors/common/nonceSelector';
import {getTrainsSearchSortInfo} from 'selectors/trains/genericSearch/sort/trainsSearchSortInfoSelector';
import {getTrainsOriginalSearchFilters} from 'selectors/trains/genericSearch/filters/getTrainsOriginalSearchFilters';

import getGenericSearchMeta from 'projects/trains/lib/meta/searchMeta/getGenericSearchMeta';
import {
    isTodaySearch,
    isTomorrowSearch,
} from 'projects/trains/lib/search/isNamedDaySearch';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import DocumentMeta from 'components/DocumentMeta/DocumentMeta';
import SocialSharingMeta from 'components/SocialSharingMeta/SocialSharingMeta';

import EnvironmentConfigContext from 'contexts/EnvironmentConfigContext';

interface ITrainsGenericSearchMetaProps {}

export const TrainsGenericSearchMeta: React.FC<ITrainsGenericSearchMetaProps> =
    () => {
        const deviceType = useDeviceType();
        const nonce = useSelector(nonceSelector);
        const {tld} = useContext(EnvironmentConfigContext);
        const context = useSelector(trainsContextSelector);
        const sort = useSelector(getTrainsSearchSortInfo);
        const filters = useSelector(getTrainsOriginalSearchFilters);

        if (!isFilledTrainsSearchContext(context)) {
            return null;
        }

        const {
            title,
            description,
            ogTitle,
            ogDescription,
            canonicalUrl,
            semanticData,
        } = getGenericSearchMeta({
            tld,
            sort,
            context,
            filters,
            deviceType,
        });

        const ogUrl = getOgUrl({canonicalUrl, context});

        return (
            <>
                <DocumentMeta
                    title={title}
                    description={description}
                    canonicalUrl={canonicalUrl}
                />
                <SocialSharingMeta
                    url={ogUrl}
                    title={ogTitle}
                    description={ogDescription}
                />
                <Helmet>
                    {semanticData.map((innerHTML, index) => (
                        <script
                            key={index}
                            type="application/ld+json"
                            nonce={nonce}
                        >
                            {innerHTML}
                        </script>
                    ))}
                </Helmet>
            </>
        );
    };

function getOgUrl({
    canonicalUrl,
    context,
}: {
    canonicalUrl: string;
    context: TrainsSearchContextType;
}): string {
    const query = {when: context.when};

    if (isTodaySearch(context) || isTomorrowSearch(context)) {
        query.when = context.originalWhen;
    }

    return `${canonicalUrl}?${qsStringify(query)}`;
}
