import React, {useMemo, useCallback, memo} from 'react';
import {useSelector} from 'react-redux';

import {isFilledTrainsSearchContext} from 'reducers/trains/context/types';
import {ETrainsGoal} from 'utilities/metrika/types/goals/trains';
import {EQueryingStatus} from 'types/trains/search/searchInfo/ITrainsSearchInfo';
import {IWithClassName} from 'types/withClassName';

import {trainsContextSelector} from 'selectors/trains/trainsContextSelector';
import nonceSelector from 'selectors/common/nonceSelector';
import {trainsDisclaimersInfoSelector} from 'selectors/trains/genericSearch/disclaimersInfo/trainsDisclaimersInfoSelector';
import {getTrainsSearchSortInfo} from 'selectors/trains/genericSearch/sort/trainsSearchSortInfoSelector';
import {getTrainsOriginalSearchInfo} from 'selectors/trains/genericSearch/search/getTrainsOriginalSearchInfo';
import {getTrainsOriginalSearchFilters} from 'selectors/trains/genericSearch/filters/getTrainsOriginalSearchFilters';

import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useCurrencies} from 'utilities/hooks/useCurrencies';
import {trainsURLs} from 'projects/trains/lib/urls';
import getTrainsSortingQuery from 'projects/trains/lib/sort/getTrainsSortingQuery';
import {reachGoal} from 'utilities/metrika';
import getFiltersQuery from 'projects/trains/lib/filters/getFiltersQuery';

import LinkBackward from 'components/LinkBackward/LinkBackward';
import SearchPageBreadcrumbs from './components/SearchPageBreadcrumbs/SearchPageBreadcrumbs';
import SearchPageFooterDisclaimers from './components/SearchPageFooterDisclaimers/SearchPageFooterDisclaimers';

import cx from './SearchPageFooter.scss';

interface ISearchPageFooterProps extends IWithClassName {}

const SearchPageFooter: React.FC<ISearchPageFooterProps> = props => {
    const {className} = props;
    const deviceType = useDeviceType();
    const context = useSelector(trainsContextSelector);
    const nonce = useSelector(nonceSelector);
    const {hasDynamicPricing} = useSelector(trainsDisclaimersInfoSelector);
    const sort = useSelector(getTrainsSearchSortInfo);
    const filters = useSelector(getTrainsOriginalSearchFilters);
    const {status} = useSelector(getTrainsOriginalSearchInfo);
    const {nationalCurrency, preferredCurrency} = useCurrencies();
    const hasCurrencyDisclaimer = nationalCurrency !== preferredCurrency;

    const backwardUrl = useMemo<string | null>(() => {
        if (
            !isFilledTrainsSearchContext(context) ||
            status !== EQueryingStatus.DONE ||
            context.returnWhen
        ) {
            return null;
        }

        return trainsURLs.getTrainsSearchUrl({
            context: {
                ...context,
                from: context.to.slug,
                to: context.from.slug,
            },
            sort: getTrainsSortingQuery(sort),
            filters: getFiltersQuery(filters),
        });
    }, [context, sort, filters, status]);

    const handleBackwardLinkClick = useCallback(() => {
        reachGoal(ETrainsGoal.LINK_BACKWARD_CLICK);
    }, []);

    if (!isFilledTrainsSearchContext(context)) {
        return null;
    }

    return (
        <div className={cx(className, 'root')}>
            {backwardUrl && (
                <div className={cx('linkBackward')}>
                    <LinkBackward
                        to={backwardUrl}
                        date={context.when}
                        onClick={handleBackwardLinkClick}
                        {...prepareQaAttributes({
                            parent: 'trainsSearchFooter',
                        })}
                    />
                </div>
            )}
            <SearchPageFooterDisclaimers
                className={cx('disclaimers')}
                hasDynamicPricing={hasDynamicPricing}
                hasCurrencyDisclaimers={hasCurrencyDisclaimer}
                deviceType={deviceType}
                {...prepareQaAttributes({
                    parent: 'trainsSearchFooter',
                })}
            />
            <SearchPageBreadcrumbs
                sort={sort}
                nonce={nonce}
                context={context}
                filters={filters}
                {...prepareQaAttributes('trainsBreadcrumbs')}
            />
        </div>
    );
};

export default memo(SearchPageFooter);
