import {
    RUSSIAN_PASSPORT,
    RUSSIAN_INTERNATIONAL_PASSPORT,
    BIRTH_CERTIFICATE,
    SAILOR_PASSPORT,
    MILITARY_CARD,
    DOCUMENT_OTHER,
} from 'constants/document/documentTypes';
import {RUS, ENG} from 'constants/form/masks';
import {EFieldName} from 'components/BookingPassengerForm/constants/fieldNames';

import {EValidationType} from 'types/common/validation/validation';
import {TField} from 'types/common/validation/form';

import * as i18nBlock from 'i18n/trains-order-fields-errors';

import {WHITE_SPACE} from './common';

export const documentNumberValidation: TField = {
    name: EFieldName.documentNumber,
    validation: {
        submit: [
            {
                type: EValidationType.required,
                params: true,
                errorMessage: i18nBlock.emptyDashDocumentDashNumber(),
            },
        ],
    },
    dependentValidations: [
        /* Паспорт РФ */
        {
            conditions: [
                {
                    path: {fieldName: EFieldName.documentType},
                    value: [
                        {
                            type: EValidationType.oneOf,
                            params: [RUSSIAN_PASSPORT],
                        },
                    ],
                },
            ],
            validation: {
                submit: [
                    {
                        type: EValidationType.regex,
                        params: '^\\d{2}\\d{2}\\d{6}$',
                        errorMessage:
                            i18nBlock.incorrectDashRussianDashPassportDashNumber(),
                    },
                    {
                        type: EValidationType.regex,
                        params: '^\\d{2}((9[789])|([01]\\d)|(2[0123]))',
                        errorMessage:
                            i18nBlock.incorrectDashRussianDashPassportDashSeriesDashNumber(),
                    },
                ],
            },
        },
        /* Загран РФ */
        {
            conditions: [
                {
                    path: {fieldName: EFieldName.documentType},
                    value: [
                        {
                            type: EValidationType.oneOf,
                            params: [RUSSIAN_INTERNATIONAL_PASSPORT],
                        },
                    ],
                },
            ],
            validation: {
                submit: [
                    {
                        type: EValidationType.regex,
                        params: '^\\d{9}$',
                        errorMessage:
                            i18nBlock.incorrectDashRussianDashInternationalDashPassportDashNumber(),
                    },
                ],
            },
        },
        /* Свидетельство о рождении */
        {
            conditions: [
                {
                    path: {fieldName: EFieldName.documentType},
                    value: [
                        {
                            type: EValidationType.oneOf,
                            params: [BIRTH_CERTIFICATE],
                        },
                    ],
                },
            ],
            validation: {
                submit: [
                    {
                        type: EValidationType.regex,
                        params: {
                            regex: `^(${WHITE_SPACE})*((I{1,3}|IV|VI{0,3}|IX)|(X{1,2}(IX|IV|V?I{0,3}))|(XXXI{0,3}|XXXI?V))[${RUS}]{2}\\d{6}(${WHITE_SPACE})*$`,
                            flags: 'i',
                        },
                        errorMessage:
                            i18nBlock.incorrectDashBirthDashCertificateDashNumber(),
                    },
                ],
            },
        },
        /* Военник */
        {
            conditions: [
                {
                    path: {fieldName: EFieldName.documentType},
                    value: [
                        {
                            type: EValidationType.oneOf,
                            params: [MILITARY_CARD],
                        },
                    ],
                },
            ],
            validation: {
                submit: [
                    {
                        type: EValidationType.regex,
                        params: `^(${WHITE_SPACE})*[${RUS}]{2}\\d{6,7}(${WHITE_SPACE})*$`,
                        errorMessage:
                            i18nBlock.incorrectDashMilitaryDashCardDashNumber(),
                    },
                ],
            },
        },
        /* Паспорт моряка */
        {
            conditions: [
                {
                    path: {fieldName: EFieldName.documentType},
                    value: [
                        {
                            type: EValidationType.oneOf,
                            params: [SAILOR_PASSPORT],
                        },
                    ],
                },
            ],
            validation: {
                submit: [
                    {
                        type: EValidationType.regex,
                        params: `^(\\d{7})$`,
                        errorMessage: i18nBlock.incorrectSailorPassportNumber(),
                    },
                ],
            },
        },
        /* Другой документ */
        {
            conditions: [
                {
                    path: {fieldName: EFieldName.documentType},
                    value: [
                        {
                            type: EValidationType.oneOf,
                            params: [DOCUMENT_OTHER],
                        },
                    ],
                },
            ],
            validation: {
                submit: [
                    {
                        type: EValidationType.maxLength,
                        params: 16,
                        errorMessage:
                            i18nBlock.incorrectDashForeignDashDocumentDashNumber(),
                    },
                    {
                        type: EValidationType.regex,
                        params: `(^(([^-].*)*\\d(.*[^-])*){1,}$)|(^[${ENG}]{5,}$)`,
                        errorMessage:
                            i18nBlock.incorrectDashForeignDashDocumentDashNumber(),
                    },
                ],
            },
        },
    ],
};
