import {IGenericOrderInfo} from 'server/api/GenericOrderApi/types/common/IGenericOrderInfo';
import {EHttpStatusCodes} from 'types/httpStatusCodes';
import EGenericOrderState from 'server/api/GenericOrderApi/types/common/EGenericOrderState';

import {delay} from 'utilities/async/delay';
import {retry} from 'utilities/async/retry';
import {isUnknownAxiosError} from 'utilities/error';

import {genericOrderBrowserProvider} from 'serviceProvider/genericOrder/genericOrderBrowserProvider';

const retryOptions = {
    attempts: 3,
    delay: 3000,
};

/**
 * Дожидаемся когда места будут разбронированы
 * Ограничиваем количество попыток, т.к. может быть какая-то проблема
 * на стороне партнера и мы просто не сможем показать страницу выбора мест
 */
export async function cancelOrder(orderInfo: IGenericOrderInfo): Promise<void> {
    const {id: orderId} = orderInfo;

    if (orderInfo.state === EGenericOrderState.CANCELLED) {
        return;
    }

    try {
        await genericOrderBrowserProvider.cancelOrder({orderId});
    } catch (e) {
        // Заказ уже был отменен
        if (
            isUnknownAxiosError(e) &&
            e?.response?.status === EHttpStatusCodes.Conflict
        ) {
            return;
        }

        throw e;
    }

    await delay(retryOptions.delay);

    await retry(async () => {
        const orderState = await genericOrderBrowserProvider.getOrderState({
            orderId,
        });

        if (orderState.state !== EGenericOrderState.CANCELLED) {
            throw new Error(`Order "${orderId}" is not cancelled!`);
        }
    }, retryOptions)();
}
