import {
    IOrderPassenger,
    IOrderTicket,
    IRefundPayment,
    ITrainsOrderDetails,
} from 'reducers/trains/order/types';
import {IGenericOrderInfo} from 'server/api/GenericOrderApi/types/common/IGenericOrderInfo';
import EDisplayOrderState from 'types/common/orders/EDisplayOrderState';
import EGenericOrderState from 'server/api/GenericOrderApi/types/common/EGenericOrderState';
import EServiceType from 'server/api/GenericOrderApi/types/common/EServiceType';
import {EDirection} from 'types/common/EDirection';
import EServiceState from 'server/api/GenericOrderApi/types/common/EServiceState';
import ITrainPassenger from 'server/api/GenericOrderApi/types/common/service/ITrainServiceInfo/ITrainPassenger/ITrainPassenger';
import ITrainTicket from 'server/api/GenericOrderApi/types/common/service/ITrainServiceInfo/ITrainPassenger/ITrainTicket';
import ERefundPartType from 'server/api/GenericOrderApi/types/common/refund/ERefundPartType';
import ERefundPartState from 'server/api/GenericOrderApi/types/common/refund/ERefundPartState';
import ERefundState from 'server/api/GenericOrderApi/types/common/refund/ERefundState';
import ERefundType from 'server/api/GenericOrderApi/types/common/refund/ERefundType';
import {TrainsOrderRefundType} from 'server/api/TrainsBookingApi/types/ITrainsOrderInfoResponse';
import EFiscalReceiptType from 'server/api/GenericOrderApi/types/common/EFiscalReceiptType';

import {CurrencyType} from 'utilities/currency/CurrencyType';
import IPrice from 'utilities/currency/PriceInterface';

/**
 * Собираем из заказа старого образца (train-api) заказ нового вида Generic
 * чтобы отображать старые заказы
 */
export default function convertTrainApiOrderToGenericOrder(
    order: ITrainsOrderDetails,
): IGenericOrderInfo {
    const {
        uid,
        userInfo,
        orderNumber,
        passengers,
        stationFrom,
        stationTo,
        carType,
        carNumber,
        coachOwner,
        specialNotice,
        twoStorey,
        partner,
        departure,
        arrival,
        insuranceStatus,
        trainTitle = '',
        brandTitle = '',
        isSuburban,
        startStation,
        endStation,
        trainTicketNumber,
        trainNumber,
        rebookingAvailable,
        paymentReceiptUrl,
        refundPayments,
    } = order;

    const totalPriceAmount = passengers.reduce((passengersTotal, passenger) => {
        return (
            passengersTotal +
            passenger.tickets.reduce((ticketsTotal, ticket) => {
                return ticketsTotal + ticket.amount;
            }, 0)
        );
    }, 0);
    const totalPrice: IPrice = {
        value: totalPriceAmount,
        currency: CurrencyType.RUB,
    };

    return {
        id: uid,
        displayState: EDisplayOrderState.FULFILLED,
        orderPriceInfo: {
            price: totalPrice,
            originalPrice: totalPrice,
            discountAmount: {value: 0, currency: CurrencyType.RUB},
            promoCampaigns: {},
            codeApplicationResults: [],
        },
        payment: {
            paymentUrl: '',
            errorInfo: null,
            receipts: paymentReceiptUrl
                ? [{url: paymentReceiptUrl, type: EFiscalReceiptType.ACQUIRE}]
                : [],
        },
        prettyId: orderNumber || '',
        refundPartInfo: null,
        reloadOrderAt: null,
        services: [
            {
                id: 'service-id',
                serviceType: EServiceType.TRAIN,
                state: EServiceState.CONFIRMED,
                refundPartInfo: null,
                trainInfo: {
                    error: null,
                    insuranceStatus,

                    direction: EDirection.FORWARD,
                    segmentIndex: 0,

                    partnerOrderId: 0,
                    reservationNumber: undefined,
                    stationFrom,
                    stationTo,
                    trainInfo: {
                        trainTitle,
                        startStationTitle: startStation,
                        endStationTitle: endStation,
                        startSettlementTitle: '',
                        endSettlementTitle: '',
                        brandTitle,
                        suburban: isSuburban,
                        trainTicketNumber,
                        trainNumber,
                    },
                    carType,
                    compartmentGender: null,
                    arrival,
                    departure,
                    partner,
                    carNumber,
                    specialNotice: specialNotice || '',
                    warnings: [],
                    twoStorey,
                    coachOwner,
                    passengers: passengers.map(passenger => {
                        return convertPassenger(passenger, refundPayments);
                    }),
                    rebookingAvailable,
                },
            },
        ],
        state: passengers.every(passenger => passenger.tickets[0].refund)
            ? EGenericOrderState.REFUNDED
            : EGenericOrderState.CONFIRMED,
        contactInfo: userInfo,
    };
}

function convertPassenger(
    passenger: IOrderPassenger,
    refundPayments: IRefundPayment[],
): ITrainPassenger {
    const {
        tickets,
        customerId,
        isNonRefundableTariff,
        insurance,
        ...restPassenger
    } = passenger;

    const ticket = tickets[0];

    return {
        ...restPassenger,
        nonRefundableTariff: isNonRefundableTariff,
        customerId: Number(customerId),
        ticket: convertTicket(ticket),
        insurance:
            insurance.ordered && !insurance.refunded
                ? {
                      compensation: {
                          value: insurance.compensation,
                          currency: CurrencyType.RUB,
                      },
                      amount: {
                          value: insurance.amount,
                          currency: CurrencyType.RUB,
                      },
                  }
                : null,
        refundPartInfo: ticket.refund
            ? {
                  type: ERefundPartType.SERVICE_PART,
                  state: ERefundPartState.REFUNDED,
                  context: {info: 'blank'},
                  refund: {
                      id: 'refund-id',
                      state: ERefundState.REFUNDED,
                      type: ticket.refund.type
                          ? convertRefundType(ticket.refund.type)
                          : ERefundType.USER_REFUND,
                      refundAmount: {
                          value: ticket.refund.amount,
                          currency: CurrencyType.RUB,
                      },
                      paymentRefundReceiptUrls: refundPayments
                          .filter(refundPayment =>
                              refundPayment.blankIds.includes(ticket.blankId),
                          )
                          .map(
                              refundPayment =>
                                  refundPayment.paymentRefundReceiptUrl,
                          ),
                      refundBlankToken: '',
                  },
              }
            : null,
        total: {
            value: ticket.amount,
            currency: CurrencyType.RUB,
        },
    };
}

function convertTicket(ticket: IOrderTicket): ITrainTicket {
    const {
        amount,
        blankId,
        payment,
        places,
        placesType,
        rawTariffTitle,
        tariffInfo,
        ...restTicket
    } = ticket;

    return {
        ...restTicket,
        tariffInfo,
        amount: {
            value: amount,
            currency: CurrencyType.RUB,
        },
        blankId: Number(blankId),
        canChangeElectronicRegistrationTill: null,
        canReturnTill: null,
        discountDenied: false,
        payment: {
            amount: {value: payment.amount, currency: CurrencyType.RUB},
            fee: {value: payment.fee, currency: CurrencyType.RUB},
            beddingAmount: {
                value: payment.beddingAmount,
                currency: CurrencyType.RUB,
            },
        },
        places: places.map(place => {
            return {
                number: place,
                type: null,
                typeText: placesType.description,
            };
        }),
        rawTariffTitle: tariffInfo?.title || rawTariffTitle,
    };
}

function convertRefundType(type: TrainsOrderRefundType): ERefundType {
    switch (type) {
        case TrainsOrderRefundType.USER_REFUND:
            return ERefundType.USER_REFUND;
        case TrainsOrderRefundType.OFFICE_REFUND:
            return ERefundType.TRAIN_OFFICE_REFUND;
        case TrainsOrderRefundType.INSURANCE_AUTO_RETURN:
            return ERefundType.TRAIN_INSURANCE_AUTO_RETURN;
    }
}
