import moment from 'moment';

import {TRAINS_DOCUMENT_TYPES} from 'projects/trains/constants/documentTypes';

import {
    IStoreTrainTraveler,
    ITrainOrderContacts,
    ITrainPassenger,
} from 'reducers/trains/order/types';
import ITrainsCreateOrderPassenger from 'server/api/GenericOrderApi/types/common/ITrainsCreateOrderPassenger';

import {replaceSpecialSymbols} from 'projects/trains/lib/order/fields/document';
import {sanitizeName} from 'projects/trains/lib/order/fields/masks';
import {HUMAN_DATE_RU, ROBOT} from 'utilities/dateUtils/formats';
import preparePassengerContacts from './utilities/preparePassengerContacts/preparePassengerContacts';

export default function prepareCreateOrderPassengers({
    passengersData,
    traveler,
    contacts,
}: {
    passengersData: ITrainPassenger[];
    traveler: IStoreTrainTraveler;
    contacts: ITrainOrderContacts;
}): ITrainsCreateOrderPassenger[] {
    return passengersData.map(
        (
            passenger: ITrainPassenger,
            passengerIndex: number,
        ): ITrainsCreateOrderPassenger => {
            const documentType = passenger.passengerDocument.type
                .value as TRAINS_DOCUMENT_TYPES;
            const documentValidDate =
                passenger.passengerDocument.validDate.value;

            let documentNumber = replaceSpecialSymbols(
                passenger.passengerDocument.number.value as string,
                documentType,
            );

            // @see https://st.yandex-team.ru/TRAINS-1575
            if (documentType === TRAINS_DOCUMENT_TYPES.BIRTH_CERTIFICATE) {
                documentNumber = documentNumber.toUpperCase();
            }

            return {
                index: passengerIndex,
                firstName: sanitizeName(passenger.firstName.value as string),
                lastName: sanitizeName(passenger.lastName.value as string),
                patronymic: sanitizeName(passenger.patronymic.value as string),
                birthDate: moment(
                    passenger.birthDate.value,
                    HUMAN_DATE_RU,
                ).format(ROBOT),
                docId: documentNumber,
                docValidDate: documentValidDate
                    ? moment(documentValidDate, HUMAN_DATE_RU).format(ROBOT)
                    : undefined,
                ...preparePassengerContacts({
                    storePassenger: passenger,
                    traveler,
                    contacts,
                }),
            };
        },
    );
}
