import {
    IStoreTrainTraveler,
    IStoreTrainTravelerPassenger,
    ITrainOrderContacts,
    ITrainOrderField,
    ITrainPassenger,
    ITrainPassengerContacts,
} from 'reducers/trains/order/types';

import {
    EMAIL_MASK,
    PHONE_NUMBER_MASK,
} from 'projects/trains/lib/order/fields/masks';
import areTravelerAndStorePassengersEqual from 'projects/trains/lib/order/traveler/areTravelerAndStorePassengersEqual';

export const isInvalid = (field: ITrainOrderField, mask: RegExp): boolean =>
    Boolean(field.value && !mask.test(field.value.trim()));

function buildContacts(
    passenger: IStoreTrainTravelerPassenger,
): ITrainPassengerContacts {
    const {phone, email} = passenger;

    return {
        ...(phone && {phone}),
        ...(email && {email}),
    };
}

export default function preparePassengerContacts({
    storePassenger,
    traveler,
    contacts,
}: {
    storePassenger: ITrainPassenger;
    traveler: IStoreTrainTraveler;
    contacts: ITrainOrderContacts;
}): ITrainPassengerContacts {
    const {passengers: travelerPassengers} = traveler;

    const emailOrPhone = storePassenger.emailOrPhone;

    if (!contacts.useContacts.value && emailOrPhone.value) {
        if (!isInvalid(emailOrPhone, EMAIL_MASK)) {
            return {
                email: emailOrPhone.value,
            };
        }

        if (!isInvalid(emailOrPhone, PHONE_NUMBER_MASK)) {
            return {
                phone: emailOrPhone.value,
            };
        }
    }

    for (const passenger of travelerPassengers) {
        for (const document of passenger.documents) {
            const isActive = areTravelerAndStorePassengersEqual({
                travelerPassenger: passenger,
                travelerDocument: document,
                storePassenger,
            });

            if (isActive && passenger.train_notifications_enabled) {
                return buildContacts(passenger);
            }
        }
    }

    return {
        email: contacts.email.value,
        phone: contacts.phone.value,
    };
}
