import _pickBy from 'lodash/pickBy';
import _isEmpty from 'lodash/isEmpty';

import {
    ADDITIONAL_REQUIREMENTS,
    REQUIREMENTS_IRRELEVANT_OPTION,
} from 'projects/trains/constants/requirements';
import {ETrainsPartner} from 'projects/trains/constants/trainPartners';
import {GENDER_TYPE} from 'projects/trains/constants/genders';

import {
    ITrainPassenger,
    ITrainsCoach,
    ITrainsDetails,
    ITrainsRequirements,
    TrainsPassengersCount,
} from 'reducers/trains/order/types';
import {ITrainsSaveOfferServiceParams} from 'server/services/TrainsService/types/ITrainsSaveOfferService';
import {EDirection} from 'types/common/EDirection';
import {TTrainsStoreOrderSegment} from 'projects/trains/lib/segments/types';
import {ITrainPriceDetails} from 'projects/trains/lib/order/getTrainPriceDetails/types/ITrainPriceDetails';

import getOrderGender from 'projects/trains/lib/order/gender/getOrderGender';
import getSchemaByCoach from 'projects/trains/lib/order/getSchemaByCoach';
import {restoreYtpReferer} from 'utilities/ytpReferer';
import patchRequirements from 'projects/trains/lib/api/utilities/prepareSaveOfferParams/utilitites/patchRequirements';
import {formatCoachNumber} from 'projects/trains/lib/ufs';
import prepareSaveOfferPassengers from 'projects/trains/lib/api/utilities/prepareSaveOfferParams/utilitites/prepareSaveOfferPassengers';
import getCompanies from 'projects/trains/lib/segments/features/getCompanies';

interface IPrepareSaveOfferParams {
    passengersData: ITrainPassenger[];
    trainDetails: ITrainsDetails | null;
    direction: EDirection;
    index: number;
    orderPlaces: number[];
    beddingOption: boolean;
    requirements: ITrainsRequirements | undefined;
    segment: TTrainsStoreOrderSegment | null;
    coach: ITrainsCoach | null;
    priceDetails: ITrainPriceDetails;
    gender: GENDER_TYPE | null;
    passengers: TrainsPassengersCount;
}

export function prepareSaveOfferParams({
    passengersData,
    direction,
    index,
    orderPlaces,
    beddingOption,
    requirements,
    segment,
    coach,
    priceDetails,
    gender,
    trainDetails,
    passengers,
}: IPrepareSaveOfferParams): ITrainsSaveOfferServiceParams {
    if (!segment || !trainDetails || !coach) {
        throw new Error('segment or trainsDetails or coach is undefined');
    }

    const {stationFrom, stationTo, title, thread} = segment;
    const companies = getCompanies(segment);
    const priceDetailsPlaces = priceDetails?.places;
    const isPlacesSelectedAtOnce = priceDetails?.isPlacesSelectedAtOnce;

    const {
        schemas,
        departure,
        ticketNumber,
        rawTrainName,
        imInitialStationName,
        imFinalStationName,
        isCppk,
        routePolicy,
        provider,
    } = trainDetails;

    const cabinGenderKind =
        getOrderGender({coach, passengers, gender}) || undefined;

    const schema = getSchemaByCoach(coach, schemas);
    const upper = schema?.placeFlags?.upper || [];
    const middle = schema?.placeFlags?.middle || [];

    const filteredRequirements = _pickBy(requirements, requirement => {
        return requirement !== REQUIREMENTS_IRRELEVANT_OPTION;
    });

    const ytpReferer = restoreYtpReferer();

    const patchedRequirements = patchRequirements({
        requirements: filteredRequirements,
        orderPlaces,
        upper,
        middle,
    });

    return {
        direction,
        segmentIndex: index,
        feeCalculationToken: coach.feeCalculationToken,
        trainNumber: coach.trainStartNumber,
        trainTicketNumber: ticketNumber,
        trainTitle: title,
        brandTitle: thread?.deluxeTrain?.shortTitle || rawTrainName,
        companyTitle: companies[0]?.content,
        rawTrainName,
        stationFromId: Number(stationFrom.id),
        stationToId: Number(stationTo.id),
        departure,
        arrival: coach.arrival,
        carNumber:
            _isEmpty(filteredRequirements) || gender || isCppk
                ? formatCoachNumber(coach.number)
                : undefined,
        carType: coach.type,
        serviceClass: coach.serviceClass.code,
        electronicRegistration: coach.electronicTicket,
        canChooseBedding: coach.canChooseBedding,
        bedding: beddingOption,
        passengers: prepareSaveOfferPassengers({
            passengersData,
            priceDetailsPlaces,
            departure,
            stationFrom,
            coach,
            isPlacesSelectedAtOnce,
        }),
        places: orderPlaces,
        partner: ETrainsPartner.IM,
        cabinGenderKind,
        requirements: {
            arrangement: patchedRequirements.arrangement,
            count: patchedRequirements.count,
            number: isNaN(Number(patchedRequirements.storey))
                ? Number(patchedRequirements.storey)
                : undefined,
        },
        additionalPlaceRequirements:
            patchedRequirements?.additional ||
            (coach.petsAllowed &&
                coach.petsSegregated &&
                ADDITIONAL_REQUIREMENTS.WITH_PETS) ||
            undefined,
        amount: priceDetails?.price?.value,
        routePolicy,
        imInitialStationName,
        imFinalStationName,
        isCppk,
        schemeId: coach.schemaId,
        provider,
        isTransfer: false,
        ytpReferer,
    };
}
