import {EDirection} from 'types/common/EDirection';
import {WHEN_SPECIAL_VALUE} from 'types/common/When';
import {TQueryTypeWithArray} from 'types/common/IQueryParams';
import {TTrainsSegmentId} from 'types/trains/common/segment/ITrainsSegment';

import {getWhenFromSearchUrlParams} from 'projects/trains/lib/context/getWhenFromSearchUrlParams';
import {
    convertPathDirectionToDirection,
    TPathDirection,
} from 'projects/trains/lib/urls/pathDirectionHelpers';

export interface ITrainsContextFromSearchUrlParams {
    from: {
        slug: string;
    };
    to: {
        slug: string;
    };
    when: WHEN_SPECIAL_VALUE | string;
    originalWhen: WHEN_SPECIAL_VALUE | string;
    returnWhen?: string;
    direction?: EDirection;
    forwardSegmentId?: TTrainsSegmentId;
    backwardSegmentId?: TTrainsSegmentId;
}

/**
 * Возвращает жд контекст из поисковых параметров url'а для страницы жд поиска.
 *
 * @param context - упрощенный контекст
 * @param context.fromSlug - идентификатор пункта отправления
 * @param context.toSlug - идентификатор пункта прибытия
 * @param context.when - дата или идентификатор даты (today, tomorrow)
 */
export default function getContextFromSearchUrlParams({
    fromSlug,
    toSlug,
    when,
    returnWhen,
    direction,
    forwardSegmentId,
    backwardSegmentId,
}: {
    fromSlug: string;
    toSlug: string;
    when: TQueryTypeWithArray;
    returnWhen?: TQueryTypeWithArray;
    direction?: TQueryTypeWithArray;
    forwardSegmentId?: TQueryTypeWithArray;
    backwardSegmentId?: TQueryTypeWithArray;
}): ITrainsContextFromSearchUrlParams {
    return {
        from: {
            slug: fromSlug,
        },
        to: {
            slug: toSlug,
        },
        when: getWhenFromSearchUrlParams(when),
        originalWhen: when ? String(when) : WHEN_SPECIAL_VALUE.ALL_DAYS,
        returnWhen: returnWhen ? String(returnWhen) : undefined,
        direction: convertPathDirectionToDirection(direction as TPathDirection),
        forwardSegmentId: forwardSegmentId
            ? String(forwardSegmentId)
            : undefined,
        backwardSegmentId: backwardSegmentId
            ? String(backwardSegmentId)
            : undefined,
    };
}
