import {Duration} from 'moment';

import {
    EShorteningOption,
    ETimeUnit,
    TIME_UNITS_LENGTHS,
    TIME_UNITS_LIST,
} from 'projects/trains/lib/date/duration/constants';

import getUnitText from 'projects/trains/lib/date/duration/getUnitText';

interface IFormatDurationOptions {
    shortestUnit?: ETimeUnit;
    longestUnit?: ETimeUnit;
    short?: EShorteningOption;
}

export default function formatDuration(
    duration: Duration,
    opts: IFormatDurationOptions = {},
): string {
    const {
        shortestUnit = ETimeUnit.MINUTES,
        longestUnit = ETimeUnit.WEEKS,
        short = EShorteningOption.AUTO,
    } = opts;

    let index = TIME_UNITS_LIST.indexOf(longestUnit);
    const lastIndex = TIME_UNITS_LIST.indexOf(shortestUnit);

    let rest = duration.asMilliseconds();

    const units: [ETimeUnit, number][] = [];

    while (index >= lastIndex && rest > 0) {
        const unit = TIME_UNITS_LIST[index];

        const unitLength = TIME_UNITS_LENGTHS[unit];
        const unitCount = Math.floor(rest / unitLength);

        if (unitCount > 0) {
            units.push([unit, unitCount]);
            rest -= unitCount * unitLength;
        }

        index--;
    }

    const isShort = getIsShort(short, units.length);

    return units
        .map(([unit, count]) => getUnitText(count, unit, isShort))
        .join(' ');
}

function getIsShort(
    shorteningOption: EShorteningOption,
    unitsCount: number,
): boolean {
    if (shorteningOption === EShorteningOption.ALWAYS) {
        return true;
    }

    if (shorteningOption === EShorteningOption.NEVER) {
        return false;
    }

    return unitsCount > 1;
}
