import moment, {Duration} from 'moment';

import {
    ETimeUnit,
    TIME_UNITS_LENGTHS,
    TIME_UNITS_LIST,
} from 'projects/trains/lib/date/duration/constants';

export interface IRoundDurationOptions {
    shortestUnit?: ETimeUnit;
    longestUnit?: ETimeUnit;
    maxGaps?: number;
    maxUnits?: number;
}

export function roundDuration(
    duration: Duration,
    options: IRoundDurationOptions = {},
): Duration {
    const {
        shortestUnit = ETimeUnit.MINUTES,
        longestUnit = ETimeUnit.WEEKS,
        maxUnits = 2,
        maxGaps = 0,
    } = options;

    let unit: ETimeUnit | undefined;
    let gapsLeft = maxGaps;
    let unitsLeft = maxUnits;
    let index = TIME_UNITS_LIST.indexOf(longestUnit);
    const lastIndex = TIME_UNITS_LIST.indexOf(shortestUnit);

    let firstUnitFound = false;
    let rest = duration.asMilliseconds();

    if (rest <= 0) {
        return moment.duration();
    }

    while (rest > 0 && gapsLeft >= 0 && unitsLeft > 0 && index >= lastIndex) {
        unit = TIME_UNITS_LIST[index];

        const unitLength = TIME_UNITS_LENGTHS[unit];
        const unitCount = Math.floor(rest / unitLength);

        firstUnitFound = firstUnitFound || unitCount > 0;

        if (firstUnitFound) {
            if (unitCount > 0) {
                unitsLeft--;
                rest -= unitLength * unitCount;
            } else {
                gapsLeft--;
            }
        }

        index--;
    }

    if (!unit) {
        return moment.duration();
    }

    return moment.duration(Math.round(duration.as(unit)), unit);
}
