import {EOrderErrorType} from 'projects/trains/constants/orderErrors';

import * as i18nBlock from 'i18n/trains-error-page-order';

/**
 * Обрабатываемые коды ошибок УФС.
 */
export enum EUFSErrorCodes {
    // Статус поезда
    /**
     * До отправления поезда осталось меньше {0} часа(-ов)
     */
    TRAIN_DEPARTS_SOON = 5414,

    // Места
    /**
     * На заданном направлении (или поезде) мест нет
     */
    NO_PLACES_ON_DIRECTION = 5060,
    /**
     * По данным требованиям мест нет, попробуйте изменить требования к местам
     */
    NO_PLACES_FOR_GIVEN_PARAMETERS = 5068,
    /**
     * В выбранном вагоне свободных мест нет
     */
    NO_PLACES_IN_SELECTED_CAR = 5371,
    /**
     * Нет в одном купе
     */
    NO_PLACES_IN_SAME_COMPARTMENT = 5382,
    /**
     * Нижних нет или у окна
     */
    NO_PLACES_BOTTOM_OR_NEAR_WINDOW = 5383,
    /**
     * Только боковые
     */
    ONLY_SIDE_PLACES = 5403,

    // Бонусные карты
    /**
     * Карты лояльности применяются при оформлении билетов только на взрослых пассажиров!
     */
    LOYALTY_CARDS_ADULT_PASSENGERS_ONLY = 5301,
    /**
     * Карта применяется только при покупке билета пассажиром, на которого она оформлена!
     */
    PASSENGERS_DATA_DOES_NOT_CORRESPOND_TO_CARDHOLDER_DATA_1 = 5302,
    /**
     * Карта не принадлежит данному пассажиру!
     */
    PASSENGERS_DATA_DOES_NOT_CORRESPOND_TO_CARDHOLDER_DATA_2 = 5303,
    /**
     * ФИО пассажира не соответствуют данным владельца карты!
     */
    PASSENGERS_DATA_DOES_NOT_CORRESPOND_TO_CARDHOLDER_DATA_3 = 5308,
    /**
     * Некорректный номер карты
     */
    INCORRECT_LOYALTY_CARD_NUMBER_1 = 5306,
    /**
     * Недопустимый номер бонусной карты
     */
    INCORRECT_LOYALTY_CARD_NUMBER_2 = 5407,

    // Резервирование
    /**
     * Повторное резервирование с теми же параметрами невозможно в течение {0} минут (с момента первого заказа),
     * либо до осуществления возврата на этот заказ
     */
    REPEATED_RESERVATION = 5043,
}

/**
 * Коды ошибок УФС про места.
 */
export const PLACES_ERROR_CODES = [
    EUFSErrorCodes.NO_PLACES_ON_DIRECTION,
    EUFSErrorCodes.NO_PLACES_FOR_GIVEN_PARAMETERS,
    EUFSErrorCodes.NO_PLACES_IN_SELECTED_CAR,
    EUFSErrorCodes.NO_PLACES_IN_SAME_COMPARTMENT,
    EUFSErrorCodes.NO_PLACES_BOTTOM_OR_NEAR_WINDOW,
    EUFSErrorCodes.ONLY_SIDE_PLACES,
];

/**
 * Коды ошибок УФС про бонусные карты.
 */
export const LOYALTY_CARDS_ERROR_CODES = [
    EUFSErrorCodes.LOYALTY_CARDS_ADULT_PASSENGERS_ONLY,
    EUFSErrorCodes.PASSENGERS_DATA_DOES_NOT_CORRESPOND_TO_CARDHOLDER_DATA_1,
    EUFSErrorCodes.PASSENGERS_DATA_DOES_NOT_CORRESPOND_TO_CARDHOLDER_DATA_2,
    EUFSErrorCodes.PASSENGERS_DATA_DOES_NOT_CORRESPOND_TO_CARDHOLDER_DATA_3,
    EUFSErrorCodes.INCORRECT_LOYALTY_CARD_NUMBER_1,
    EUFSErrorCodes.INCORRECT_LOYALTY_CARD_NUMBER_2,
];

export interface IPartnerError {
    code: keyof typeof EOrderErrorType;
    message?: string;
    description?: string;
}

function getErrorType(error: IPartnerError | null): EOrderErrorType {
    if (!error) {
        return EOrderErrorType.COMMON;
    }

    return EOrderErrorType[error.code] ?? EOrderErrorType.COMMON;
}

function getErrorDescription(error: IPartnerError | null): string {
    return (
        error?.description ||
        error?.message ||
        i18nBlock.commonDashErrorDashDescription()
    );
}

function getErrorSolution(
    partnerError: IPartnerError | null,
): string | undefined {
    return partnerError ? undefined : i18nBlock.commonDashErrorDashSolution();
}

interface IOrderError {
    partnerError: IPartnerError | null;
    type: EOrderErrorType;
    description: string;
    solution?: string;
}

/**
 * Для страницы выбора мест. Из trainDetails может прийти ошибка.
 * @param error
 */
export function getOrderError(error: IPartnerError | null): IOrderError {
    return {
        partnerError: error,
        type: getErrorType(error),
        description: getErrorDescription(error),
        solution: getErrorSolution(error),
    };
}
