import {TIMES_OF_DAY} from 'utilities/dateUtils/constants';

import {
    ETrainsFilterType,
    ITrainsArrivalFilter,
    ITrainsFilters,
} from 'types/trains/search/filters/ITrainsFilters';
import {ETimeOfDay} from 'utilities/dateUtils/types';
import {ITrainsSearchLocation} from 'types/trains/search/ITrainsSearchLocation';
import {ITrainsTariffApiSegment} from 'server/api/TrainsApi/types/ITrainsGetTariffsApi/models';

import {getTimeOfDay} from 'utilities/dateUtils';
import {getLocalArrivalTime} from 'projects/trains/lib/segments/cachedSegmentTime';
import {isTimeOfDay} from 'utilities/dateUtils/isTimeOfDay';
import BaseListManager from 'projects/trains/lib/filters/managers/utilities/BaseListManager';

export class Arrival extends BaseListManager<
    ETimeOfDay,
    ETimeOfDay,
    ETrainsFilterType.ARRIVAL
> {
    type: ETrainsFilterType.ARRIVAL = ETrainsFilterType.ARRIVAL;

    apply(value: ETimeOfDay[] = [], segment: ITrainsTariffApiSegment): boolean {
        if (!value.length) {
            return true;
        }

        const arrivalTime = getTimeOfDay(getLocalArrivalTime(segment).hour());

        return Boolean(arrivalTime && value.includes(arrivalTime));
    }

    updateOptions(
        options: ETimeOfDay[],
        segment: ITrainsTariffApiSegment,
    ): ETimeOfDay[] {
        // небольшая оптимизация, т.к. дальнейшие вызовы функции не изменят опции фильтра
        if (options.length === TIMES_OF_DAY.length) {
            return options;
        }

        const time = getTimeOfDay(getLocalArrivalTime(segment).hour());

        if (time && !options.includes(time)) {
            options.push(time);
        }

        return options;
    }

    /**
     * Возвращает список опций из данных обо всех фильтрах (из state).
     */
    getOptionsFromFiltersData(filtersData: ITrainsFilters): ETimeOfDay[] {
        return filtersData[this.type].options;
    }

    /**
     * Возвращает начальное состояние данного фильтра.
     */
    initFilterData(segments: ITrainsTariffApiSegment[]): ITrainsArrivalFilter {
        const options = this.getOptions(segments);
        const availableWithOptions = this.isAvailableWithOptions(options);
        const filteredSegmentIndices: boolean[] = new Array(
            segments.length,
        ).fill(true);

        return {
            value: this.getDefaultValue(),
            options,
            activeOptions: options,
            availableWithOptions,
            availableWithActiveOptions: availableWithOptions,
            type: this.type,
            filteredSegmentIndices,
        };
    }

    deserializeFromQuery({arrival}: ITrainsSearchLocation): ETimeOfDay[] {
        if (!arrival) {
            return [];
        }

        const optionsList: string[] = Array.isArray(arrival)
            ? arrival
            : [arrival];

        return optionsList.filter(isTimeOfDay);
    }

    serializeToQuery(arrival: ETimeOfDay[]): {arrival: ETimeOfDay[]} {
        return {arrival};
    }

    formatOptions(
        options: ETimeOfDay[],
        formatTextFn?: (
            option: {value: string; text: string},
            index: number,
        ) => string,
    ): {value: string; text: string}[] {
        const formattedOptions = options.map(value => ({
            value,
            text: value,
        }));

        return formatTextFn
            ? this.formatTextWithFn(formattedOptions, formatTextFn)
            : formattedOptions;
    }
}

export default new Arrival();
