import {TIMES_OF_DAY} from 'utilities/dateUtils/constants';

import {
    ETrainsFilterType,
    ITrainsDepartureFilter,
    ITrainsFilters,
} from 'types/trains/search/filters/ITrainsFilters';
import {ETimeOfDay} from 'utilities/dateUtils/types';
import {ITrainsSearchLocation} from 'types/trains/search/ITrainsSearchLocation';
import {ITrainsTariffApiSegment} from 'server/api/TrainsApi/types/ITrainsGetTariffsApi/models';

import getSegmentDepartureTimeOfDay from 'projects/trains/lib/segments/getSegmentDepartueTimeOfDay';
import {isTimeOfDay} from 'utilities/dateUtils/isTimeOfDay';
import BaseListManager from 'projects/trains/lib/filters/managers/utilities/BaseListManager';

export class Departure extends BaseListManager<
    ETimeOfDay,
    ETimeOfDay,
    ETrainsFilterType.DEPARTURE
> {
    type: ETrainsFilterType.DEPARTURE = ETrainsFilterType.DEPARTURE;

    apply(value: ETimeOfDay[], segment: ITrainsTariffApiSegment): boolean {
        if (!value.length) {
            return true;
        }

        const departureTime = getSegmentDepartureTimeOfDay(segment);

        return value.includes(departureTime);
    }

    updateOptions(
        options: ETimeOfDay[],
        segment: ITrainsTariffApiSegment,
    ): ETimeOfDay[] {
        // небольшая оптимизация, т.к. дальнейшие вызовы функции не изменят опции фильтра
        if (options.length === TIMES_OF_DAY.length) {
            return options;
        }

        const time = getSegmentDepartureTimeOfDay(segment);

        if (!options.includes(time)) {
            options.push(time);
        }

        return options;
    }

    /**
     * Возвращает список опций из данных обо всех фильтрах (из state).
     */
    getOptionsFromFiltersData(filtersData: ITrainsFilters): ETimeOfDay[] {
        return filtersData[this.type].options;
    }

    /**
     * Возвращает начальное состояние данного фильтра.
     */
    initFilterData(
        segments: ITrainsTariffApiSegment[],
    ): ITrainsDepartureFilter {
        const options = this.getOptions(segments);
        const availableWithOptions = this.isAvailableWithOptions(options);
        const filteredSegmentIndices: boolean[] = new Array(
            segments.length,
        ).fill(true);

        return {
            value: this.getDefaultValue(),
            options,
            activeOptions: options,
            availableWithOptions,
            availableWithActiveOptions: availableWithOptions,
            type: this.type,
            filteredSegmentIndices,
        };
    }

    deserializeFromQuery({departure}: ITrainsSearchLocation): ETimeOfDay[] {
        if (!departure) {
            return [];
        }

        const optionsList: string[] = Array.isArray(departure)
            ? departure
            : [departure];

        return optionsList.filter(isTimeOfDay);
    }

    serializeToQuery(departure: ETimeOfDay[]): {departure: ETimeOfDay[]} {
        return {departure};
    }

    formatOptions(
        options: ETimeOfDay[],
        formatTextFn?: (
            option: {value: string; text: string},
            index: number,
        ) => string,
    ): {value: string; text: string}[] {
        const formattedOptions = options.map(value => ({
            value,
            text: value,
        }));

        return formatTextFn
            ? this.formatTextWithFn(formattedOptions, formatTextFn)
            : formattedOptions;
    }
}

export default new Departure();
