import {
    ETrainsFilterType,
    ITrainsFilters,
    ITrainsHideWithoutPriceFilter,
    ITrainsHideWithoutPriceOptions,
} from 'types/trains/search/filters/ITrainsFilters';
import {ITrainsSearchLocation} from 'types/trains/search/ITrainsSearchLocation';
import {ITrainsTariffApiSegment} from 'server/api/TrainsApi/types/ITrainsGetTariffsApi/models';

import hasSegmentPrice from 'projects/trains/lib/segments/hasSegmentPrice';
import hasTrainsSegmentPriceDescription from 'projects/trains/lib/segments/hasTrainsSegmentPriceDescription';
import applyFilters from 'projects/trains/lib/filters/applyFilters';

export class HideWithoutPrice {
    type: ETrainsFilterType.HIDE_WITHOUT_PRICE =
        ETrainsFilterType.HIDE_WITHOUT_PRICE;

    apply(
        hideWithoutPrice: boolean,
        segment: ITrainsTariffApiSegment,
    ): boolean {
        return (
            !hideWithoutPrice ||
            (hasSegmentPrice(segment) &&
                'tariffs' in segment &&
                !hasTrainsSegmentPriceDescription(segment.tariffs))
        );
    }

    /**
     * Проверяет должен ли применяться данный фильтр.
     *
     * @param value - текущее значение фильтра
     */
    shouldApply(value: boolean): boolean {
        return !this.isDefaultValue(value);
    }

    getDefaultValue(): boolean {
        return false;
    }

    isDefaultValue(withPriceOnly: boolean): boolean {
        return withPriceOnly === this.getDefaultValue();
    }

    getDefaultOptions(): ITrainsHideWithoutPriceOptions {
        return {withPrice: false, withoutPrice: false};
    }

    /**
     * Возвращает список доступных значений для данного фильтра,
     * с учетом результатов других фильтров.
     */
    getActiveOptions({
        filtersData,
        segments,
        filteredSegments = this.getFilteredSegments({filtersData, segments}),
    }: {
        filtersData: ITrainsFilters;
        segments: ITrainsTariffApiSegment[];
        filteredSegments?: ITrainsTariffApiSegment[];
    }): ITrainsHideWithoutPriceOptions {
        return filteredSegments.reduce(
            (options, segment) => this.updateOptions(options, segment),
            this.getDefaultOptions(),
        );
    }

    /**
     * Возвращает список сегментов, которые соответствую всем выбранным фильтрам без
     * учета данного фильтра.
     */
    getFilteredSegments({
        segments,
        filtersData,
    }: {
        segments: ITrainsTariffApiSegment[];
        filtersData: ITrainsFilters;
    }): ITrainsTariffApiSegment[] {
        return segments.reduce<ITrainsTariffApiSegment[]>(
            (filteredSegments, segment, segmentIndex) => {
                if (
                    applyFilters({
                        segmentIndex,
                        filtersData,
                        excludeFilterType: this.type,
                    })
                ) {
                    filteredSegments.push(segment);
                }

                return filteredSegments;
            },
            [],
        );
    }

    /**
     * Возвращает обновленное стостояние доступных значений фильтров.
     */
    updateActiveOptions({
        segments,
        filtersData,
    }: {
        segments: ITrainsTariffApiSegment[];
        filtersData: ITrainsFilters;
    }): ITrainsFilters {
        const filteredSegments = this.getFilteredSegments({
            segments,
            filtersData,
        });
        const activeOptions = this.getActiveOptions({
            filtersData,
            segments,
            filteredSegments,
        });
        const options = this.getOptionsFromFiltersData(filtersData);

        return {
            ...filtersData,
            [this.type]: {
                ...filtersData[this.type],
                options,
                activeOptions,
                availableWithActiveOptions:
                    this.isAvailableWithActiveOptions(activeOptions),
            },
        };
    }

    updateOptions(
        options: ITrainsHideWithoutPriceOptions,
        segment: ITrainsTariffApiSegment,
    ): ITrainsHideWithoutPriceOptions {
        if (options.withPrice && options.withoutPrice) {
            return options;
        }

        if (
            'tariffs' in segment &&
            hasTrainsSegmentPriceDescription(segment.tariffs)
        ) {
            return {
                withPrice: options.withPrice,
                withoutPrice: true,
            };
        }

        if (hasSegmentPrice(segment)) {
            return {
                withPrice: true,
                withoutPrice: options.withoutPrice,
            };
        }

        return {
            withPrice: options.withPrice,
            withoutPrice: true,
        };
    }

    /**
     * Возвращает список доступных значений данного фильтра.
     */
    getOptions(
        segments: ITrainsTariffApiSegment[],
    ): ITrainsHideWithoutPriceOptions {
        return segments.reduce(
            (prevOptions, segment) => this.updateOptions(prevOptions, segment),
            this.getDefaultOptions(),
        );
    }

    /**
     * Возвращает список опций из данных обо всех фильтрах (из state).
     */
    getOptionsFromFiltersData(
        filtersData: ITrainsFilters,
    ): ITrainsHideWithoutPriceOptions {
        return filtersData[this.type].options;
    }

    isAvailableWithOptions(options: ITrainsHideWithoutPriceOptions): boolean {
        return options.withPrice && options.withoutPrice;
    }

    /**
     * Проверяет доступность данного фильтра для переданных доступных значений.
     */
    isAvailableWithActiveOptions(
        options: ITrainsHideWithoutPriceOptions,
    ): boolean {
        return this.isAvailableWithOptions(options);
    }

    /**
     * Возвращает начальное состояние данного фильтра.
     */
    initFilterData(
        segments: ITrainsTariffApiSegment[],
    ): ITrainsHideWithoutPriceFilter {
        const options = this.getOptions(segments);
        const availableWithOptions = this.isAvailableWithOptions(options);
        const filteredSegmentIndices: boolean[] = new Array(
            segments.length,
        ).fill(true);

        return {
            value: this.getDefaultValue(),
            options,
            activeOptions: options,
            availableWithOptions,
            availableWithActiveOptions: availableWithOptions,
            type: this.type,
            filteredSegmentIndices,
        };
    }

    deserializeFromQuery({seats}: ITrainsSearchLocation): boolean {
        return typeof seats === 'string' && seats.toLowerCase() === 'y';
    }

    serializeToQuery(value: boolean): {seats: 'y' | undefined} {
        return {
            seats: this.isDefaultValue(value) ? undefined : 'y',
        };
    }

    /**
     * Возвращает массив, в котором ключом является индекс сегмента, а значение булевый признак,
     * того, что сегмент прошел фильтрацию по данному признаку.
     *
     * @param value - значение фильтра
     * @param segments - сегменты
     */
    getFilteredSegmentIndices(
        value: boolean,
        segments: ITrainsTariffApiSegment[],
    ): boolean[] {
        if (!this.shouldApply(value)) {
            return new Array(segments.length).fill(true);
        }

        return segments.map(segment => this.apply(value, segment));
    }

    setFilterValue({
        filtersData,
        value,
        segments,
    }: {
        filtersData: ITrainsFilters;
        value: boolean;
        segments: ITrainsTariffApiSegment[];
    }): ITrainsFilters {
        return {
            ...filtersData,
            [this.type]: {
                ...filtersData[this.type],
                value,
                filteredSegmentIndices: this.getFilteredSegmentIndices(
                    value,
                    segments,
                ),
            },
        };
    }
}

export default new HideWithoutPrice();
