import {
    ETrainsFilterType,
    ITrainsFilters,
    ITrainsHighSpeedTrainFilter,
    ITrainsSimpleFilterOption,
} from 'types/trains/search/filters/ITrainsFilters';
import {ITrainsSearchLocation} from 'types/trains/search/ITrainsSearchLocation';
import {ITrainsTariffApiSegment} from 'server/api/TrainsApi/types/ITrainsGetTariffsApi/models';

import BaseListManager from 'projects/trains/lib/filters/managers/utilities/BaseListManager';

function getHighSpeedTrain(
    segment: ITrainsTariffApiSegment,
): ITrainsSimpleFilterOption | null {
    const deluxeTrain =
        ('thread' in segment && segment.thread?.deluxeTrain) || null;

    return deluxeTrain?.isHighSpeed
        ? {
              value: `${deluxeTrain.id}`,
              text: deluxeTrain.title,
          }
        : null;
}

export class HighSpeedTrain extends BaseListManager<
    string,
    ITrainsSimpleFilterOption,
    ETrainsFilterType.HIGH_SPEED_TRAIN
> {
    type: ETrainsFilterType.HIGH_SPEED_TRAIN =
        ETrainsFilterType.HIGH_SPEED_TRAIN;

    apply(value: string[], segment: ITrainsTariffApiSegment): boolean {
        const train = getHighSpeedTrain(segment);

        return train ? value.includes(train.value) : false;
    }

    validateValue(
        value: string[] = [],
        options: ITrainsSimpleFilterOption[] = [],
    ): string[] {
        const trainIds = options.map(item => item.value);

        return value.filter(item => trainIds.includes(item));
    }

    updateOptions(
        options: ITrainsSimpleFilterOption[],
        segment: ITrainsTariffApiSegment,
    ): ITrainsSimpleFilterOption[] {
        const train = getHighSpeedTrain(segment);

        if (!train || options.find(({value}) => value === train.value)) {
            return options;
        }

        return [...options, train];
    }

    /**
     * Возвращает список опций из данных обо всех фильтрах (из state).
     */
    getOptionsFromFiltersData(
        filtersData: ITrainsFilters,
    ): ITrainsSimpleFilterOption[] {
        return filtersData[this.type].options;
    }

    isAvailableWithOptions(
        options: ITrainsSimpleFilterOption[],
        segments: ITrainsTariffApiSegment[],
    ): boolean {
        const hasNoHighSpeedTrain = segments.some(s => !getHighSpeedTrain(s));

        if (hasNoHighSpeedTrain) {
            return options.length > 0;
        }

        return options.length > 1;
    }

    /**
     * Возвращает начальное состояние данного фильтра.
     */
    initFilterData(
        segments: ITrainsTariffApiSegment[],
    ): ITrainsHighSpeedTrainFilter {
        const options = this.getOptions(segments);
        const availableWithOptions = this.isAvailableWithOptions(
            options,
            segments,
        );
        const filteredSegmentIndices: boolean[] = new Array(
            segments.length,
        ).fill(true);

        return {
            value: this.getDefaultValue(),
            options,
            activeOptions: options,
            availableWithOptions,
            availableWithActiveOptions: availableWithOptions,
            type: this.type,
            filteredSegmentIndices,
        };
    }

    deserializeFromQuery({highSpeedTrain}: ITrainsSearchLocation): string[] {
        if (!highSpeedTrain) {
            return this.getDefaultValue();
        }

        return Array.isArray(highSpeedTrain)
            ? [...new Set(highSpeedTrain)]
            : [highSpeedTrain];
    }

    serializeToQuery(value: string[]): {highSpeedTrain: string[]} {
        return {highSpeedTrain: value};
    }

    formatOptions(
        options: ITrainsSimpleFilterOption[],
        formatTextFn?: (
            option: ITrainsSimpleFilterOption,
            index: number,
        ) => string,
    ): ITrainsSimpleFilterOption[] {
        return formatTextFn
            ? this.formatTextWithFn(options, formatTextFn)
            : options;
    }
}

export default new HighSpeedTrain();
