import isPlaneObject from 'lodash/isPlainObject';

import {
    isTrainsCoachType,
    TRAIN_COACH_TYPE,
} from 'projects/trains/constants/coachType';

import {
    ETrainsFilterType,
    ITrainsFilters,
    ITrainsPriceRangeFilter,
    ITrainsPriceRangeOption,
} from 'types/trains/search/filters/ITrainsFilters';
import {ITrainsSearchLocation} from 'types/trains/search/ITrainsSearchLocation';
import {isNotNull, isNotUndefined} from 'types/utilities';
import {ITrainsTariffApiSegment} from 'server/api/TrainsApi/types/ITrainsGetTariffsApi/models';

import {
    getSegmentPrices,
    getTariffClassKeys,
    IPriceWithTariffName,
    isPriceWithTariffName,
} from 'projects/trains/lib/segments/tariffs';
import filterTariffClassKeysByTrainTariffClass from 'projects/trains/lib/filters/filterTariffClassKeysByTrainTariffClass';
import {CurrencyType} from 'utilities/currency/CurrencyType';
import IPrice from 'utilities/currency/PriceInterface';
import BaseListManager from 'projects/trains/lib/filters/managers/utilities/BaseListManager';

const COUNT_RANGE = 4;
const FREQUENCY_RATES: PartialRecord<CurrencyType, number> = {
    [CurrencyType.RUB]: 100,
    [CurrencyType.UAH]: 50,
};

function floorToFrequencyRate(number: number, frequencyRate: number): number {
    return Math.floor(number / frequencyRate) * frequencyRate;
}

function ceilToFrequencyRate(number: number, frequencyRate: number): number {
    return Math.ceil(number / frequencyRate) * frequencyRate;
}

export class PriceRange extends BaseListManager<
    ITrainsPriceRangeOption,
    ITrainsPriceRangeOption,
    ETrainsFilterType.PRICE_RANGE
> {
    type: ETrainsFilterType.PRICE_RANGE = ETrainsFilterType.PRICE_RANGE;

    apply(
        value: ITrainsPriceRangeOption[],
        segment: ITrainsTariffApiSegment,
    ): boolean {
        return this.getSegmentPrices(segment).some(price =>
            value.some(range => this.priceInRange(range, price.value)),
        );
    }

    /**
     * Возвращает список доступных значений для данного фильтра,
     * с учетом результатов других фильтров
     */
    getActiveOptions({
        filtersData,
        segments,
        filteredSegments = this.getFilteredSegments({segments, filtersData}),
    }: {
        filtersData: ITrainsFilters;
        segments: ITrainsTariffApiSegment[];
        filteredSegments?: ITrainsTariffApiSegment[];
    }): ITrainsPriceRangeOption[] {
        const options = this.getOptions(segments);

        // возвращаем только те опции (дипазоны цен), что соответсвуют оставшимся сегементам
        return options.filter(option =>
            filteredSegments.some(segment => {
                const prices = this.getSegmentPricesForDisplay({
                    segment,
                    filtersData,
                });

                return prices.some(({value}) =>
                    this.priceInRange(option, value),
                );
            }),
        );
    }

    /**
     * Обновляет список опций на основе данных об отфильтрованных сегментах
     */
    updateOptionsForFilteredSegments({
        options,
        filtersData,
        segments,
        filteredSegments = this.getFilteredSegments({segments, filtersData}),
    }: {
        options: ITrainsPriceRangeOption[];
        filtersData: ITrainsFilters;
        segments: ITrainsTariffApiSegment[];
        filteredSegments?: ITrainsTariffApiSegment[];
    }): ITrainsPriceRangeOption[] {
        const newOptions: ITrainsPriceRangeOption[] = [];

        // создаем копию option и обнуляем количество тарифов в диапазонах
        options.forEach(option => {
            newOptions.push({
                ...option,
                count: 0,
            });
        });

        // получаем массив со всеми ценами (которые будут показаны пользователю)
        // отфильтрованных сегментов
        const prices = filteredSegments.reduce<number[]>(
            (pricesArray, segment) =>
                pricesArray.concat(
                    this.getSegmentPricesForDisplay({segment, filtersData}).map(
                        ({value}) => value,
                    ),
                ),
            [],
        );

        // обновляем информацию о количестве тарифов в диапазонах
        prices.forEach(price => {
            newOptions.some(option => {
                if (this.priceCorrespondsToMax(price, option.max)) {
                    option.count++;

                    return true;
                }

                return false;
            });
        });

        return newOptions;
    }

    updateOptions(
        options: ITrainsPriceRangeOption[],
    ): ITrainsPriceRangeOption[] {
        return options;
    }

    /**
     * Возвращает список доступных значений данного фильтра
     */
    getOptions(segments: ITrainsTariffApiSegment[]): ITrainsPriceRangeOption[] {
        // определяем национальную валюту
        let currency = CurrencyType.RUB;

        segments.some(segment => {
            const segmentPrices = this.getSegmentPrices(segment);

            if (segmentPrices.length) {
                currency = segmentPrices[0].currency;
            }

            return currency;
        });

        const {totalCount, roundRanges} = this.getRangesByFrequencyRate(
            segments,
            currency,
        );
        // собираем все промежутки в большие интервалы
        const approximateCountInRange = Math.floor(totalCount / COUNT_RANGE);
        let ranges: {count: number; min: number; max: number}[] = [];
        let currentRange: {
            count: number;
            min: number | null;
            max: number | null;
        } = {
            count: 0,
            min: null,
            max: null,
        };

        [...roundRanges.values()]
            .sort((a, b) => a.min - b.min)
            .forEach(roundRange => {
                if (currentRange.min === null) {
                    currentRange.min = roundRange.min;
                }

                const newCount = currentRange.count + roundRange.count;
                const rangeFilled =
                    currentRange.count && newCount >= approximateCountInRange;

                if (rangeFilled && ranges.length < COUNT_RANGE - 1) {
                    if (
                        currentRange.min !== null &&
                        currentRange.max !== null
                    ) {
                        ranges.push({
                            count: currentRange.count,
                            min: currentRange.min,
                            max: currentRange.max,
                        });
                    }

                    const min = currentRange.max;

                    currentRange = {
                        count: 0,
                        min,
                        max: roundRange.max,
                    };
                }

                currentRange.count += roundRange.count;
                currentRange.max = roundRange.max;
            });

        if (currentRange.min !== null && currentRange.max !== null) {
            ranges.push({
                count: currentRange.count,
                min: currentRange.min,
                max: currentRange.max,
            });
        }

        ranges = ranges.filter(range => range.count);

        return ranges.length > 1
            ? ranges.map(range => {
                  return {
                      ...range,
                      value: `${range.min}-${range.max}`,
                      currency,
                  };
              })
            : [];
    }

    /**
     * Возвращает список опций из данных обо всех фильтрах (из state).
     */
    getOptionsFromFiltersData(
        filtersData: ITrainsFilters,
    ): ITrainsPriceRangeOption[] {
        return filtersData[this.type].options;
    }

    /**
     * Возвращает начальное состояние данного фильтра.
     */
    initFilterData(
        segments: ITrainsTariffApiSegment[],
    ): ITrainsPriceRangeFilter {
        const options = this.getOptions(segments);
        const availableWithOptions = this.isAvailableWithOptions(options);
        const filteredSegmentIndices: boolean[] = new Array(
            segments.length,
        ).fill(true);

        return {
            value: this.getDefaultValue(),
            options,
            activeOptions: options,
            availableWithOptions,
            availableWithActiveOptions: availableWithOptions,
            type: this.type,
            filteredSegmentIndices,
        };
    }

    deserializeFromQuery({
        priceRange,
    }: ITrainsSearchLocation): {min: number; max: number}[] {
        if (!priceRange) {
            return [];
        }

        const values = Array.isArray(priceRange) ? priceRange : [priceRange];

        return values
            .map(value => value.trim())
            .map(this.parseOptionFromString)
            .filter(isNotNull);
    }

    serializeToQuery(value: ITrainsPriceRangeOption[]): {priceRange: string[]} {
        return {
            priceRange: value.map(val => val.value),
        };
    }

    validateValue(
        value: ITrainsPriceRangeOption[],
        options: ITrainsPriceRangeOption[],
    ): ITrainsPriceRangeOption[] {
        if (!Array.isArray(value)) {
            return this.getDefaultValue();
        }

        return value
            .map(
                val =>
                    val &&
                    options.find(
                        option =>
                            typeof val.min !== 'undefined' &&
                            typeof val.max !== 'undefined' &&
                            val.min === option.min &&
                            val.max === option.max,
                    ),
            )
            .filter(isNotUndefined)
            .filter(Boolean);
    }

    reformatOptions(
        formattedOptions: ({min: number; max: number} | string)[],
        options: ITrainsPriceRangeOption[],
    ): ITrainsPriceRangeOption[] {
        return formattedOptions
            .map(formattedOption =>
                options.find(option => {
                    if (typeof formattedOption === 'string') {
                        return formattedOption === option.value;
                    } else if (isPlaneObject(formattedOption)) {
                        return (
                            formattedOption.min === option.min &&
                            formattedOption.max === option.max
                        );
                    }

                    return false;
                }),
            )
            .filter(isNotUndefined)
            .filter(Boolean);
    }

    setFilterValue({
        filtersData,
        value,
        segments,
    }: {
        filtersData: ITrainsFilters;
        value: {min: number; max: number}[];
        segments: ITrainsTariffApiSegment[];
    }): ITrainsFilters {
        const options = this.getOptionsFromFiltersData(filtersData);

        let validatedValue: ITrainsPriceRangeOption[];

        const min = value?.[0]?.min ?? options?.[0]?.min ?? 0;
        const max =
            value?.[0]?.max ?? options?.[options.length - 1]?.max ?? Infinity;

        const minOptions = options?.[0]?.min ?? 0;
        const maxOptions = options?.[options.length - 1]?.max ?? Infinity;

        if (min === minOptions && max === maxOptions) {
            validatedValue = [];
        } else {
            validatedValue = [
                {
                    min,
                    max,
                    count: 0,
                    currency: options?.[0]?.currency ?? CurrencyType.RUB,
                    value: `${min}-${max}`,
                },
            ];
        }

        return {
            ...filtersData,
            [this.type]: {
                ...filtersData[this.type],
                value: validatedValue,
                filteredSegmentIndices: this.getFilteredSegmentIndices(
                    validatedValue,
                    segments,
                ),
            },
        };
    }

    formatOptions(
        options: ITrainsPriceRangeOption[],
        formatText = (
            option: ITrainsPriceRangeOption,
            _index: number,
            _count: number,
        ): string => `${option.min} - ${option.max}`,
    ): {
        value: string;
        text: string;
    }[] {
        return options.map((option, index) => ({
            value: option.value,
            text: formatText(option, index, options.length),
        }));
    }

    // @ts-ignore
    formatValue(value: ITrainsPriceRangeOption[]): string[] {
        return value.map(valueItem => valueItem.value).filter(Boolean);
    }

    parseOptionFromString(value: string): {min: number; max: number} | null {
        const range = value
            .split('-')
            .map(str => parseInt(str.trim(), 10))
            .filter(number => !isNaN(number));

        if (range.length !== 2 || range[0] < 0 || range[1] <= 0) {
            return null;
        }

        return {
            min: range[0],
            max: range[1],
        };
    }

    /**
     * Проверяет соответствие цены нижней границе диапазона
     */
    priceCorrespondsToMin(price: number, min: number): boolean {
        return price >= min;
    }

    /**
     * Проверяет соответствие цены верхней границе диапазона
     */
    priceCorrespondsToMax(price: number, max: number): boolean {
        return price < max;
    }

    /**
     * Проверяет входит ли цена в диапазон
     */
    priceInRange(range: ITrainsPriceRangeOption, price: number): boolean {
        return (
            this.priceCorrespondsToMin(price, range.min) &&
            this.priceCorrespondsToMax(price, range.max)
        );
    }

    getSegmentPrices(
        segment: ITrainsTariffApiSegment,
    ): (IPrice | IPriceWithTariffName)[] {
        return getSegmentPrices(segment);
    }

    /**
     * Возвращает только те объекты с ценами, что будут показаны пользователю
     */
    getSegmentPricesForDisplay({
        segment,
        filtersData,
    }: {
        segment: ITrainsTariffApiSegment;
        filtersData: ITrainsFilters;
    }): (IPrice | IPriceWithTariffName)[] {
        const segmentPrices = this.getSegmentPrices(segment);

        let tariffClassKeys = getTariffClassKeys(segment);

        // список тарифов, подходящих под фильтр типа вагона
        tariffClassKeys = filterTariffClassKeysByTrainTariffClass(
            tariffClassKeys,
            filtersData,
        );

        // список цен, подходящий под данные тарифы
        return segmentPrices
            .filter(isPriceWithTariffName)
            .filter(
                priceObject =>
                    isTrainsCoachType(priceObject.tariffName) &&
                    tariffClassKeys.includes(priceObject.tariffName),
            );
    }

    /**
     * Возвращает массив подходящих под фильтр типов вагонов
     */
    getSuitableSegmentTariffClasses(
        value: ITrainsPriceRangeOption[],
        segment: ITrainsTariffApiSegment,
    ): TRAIN_COACH_TYPE[] {
        const prices = getSegmentPrices(segment);

        return prices
            .filter(price =>
                value.some(range => this.priceInRange(range, price.value)),
            )
            .map(price => price.tariffName);
    }

    /**
     * Возвращает минимальный шаг для диапазона цен
     */
    getFrequencyRate(currency: CurrencyType = CurrencyType.RUB): number {
        return FREQUENCY_RATES[currency] || 100;
    }

    /**
     * Возвращает количество цен в сегментах и Map, в котором цены разложены по промежуткам,
     * кратным мнимальному шагу для диапазона (frequencyRate)
     */
    getRangesByFrequencyRate(
        segments: ITrainsTariffApiSegment[],
        currency: CurrencyType = CurrencyType.RUB,
    ): {
        totalCount: number;
        roundRanges: Map<string, {min: number; max: number; count: number}>;
    } {
        const roundRanges = new Map<
            string,
            {min: number; max: number; count: number}
        >();
        let totalCount = 0;
        const frequencyRate = this.getFrequencyRate(currency);

        // раскладываем все цены на промежутки равные frequencyRate
        segments.forEach(segment =>
            this.getSegmentPrices(segment).forEach(priceObject => {
                const price = priceObject.value;
                let min = floorToFrequencyRate(price, frequencyRate);
                let max = ceilToFrequencyRate(price, frequencyRate);

                if (min === max) {
                    min -= frequencyRate;
                }

                if (price === max) {
                    // перекидываем цену в следующий промежуток
                    min += frequencyRate;
                    max += frequencyRate;
                }

                const rangeKey = `${min}-${max}`;
                const range = roundRanges.get(rangeKey) || {
                    min,
                    max,
                    count: 0,
                };

                range.count++;
                roundRanges.set(rangeKey, range);
                totalCount++;
            }),
        );

        return {
            totalCount,
            roundRanges,
        };
    }
}

export default new PriceRange();
