import {
    ETrainsFilterType,
    ITrainsFilters,
    ITrainsSimpleFilterOption,
    ITrainsStationFromFilter,
} from 'types/trains/search/filters/ITrainsFilters';
import {ITrainsSearchLocation} from 'types/trains/search/ITrainsSearchLocation';
import {ITrainsTariffApiSegment} from 'server/api/TrainsApi/types/ITrainsGetTariffsApi/models';

import BaseListManager from 'projects/trains/lib/filters/managers/utilities/BaseListManager';

export class StationFrom extends BaseListManager<
    string,
    ITrainsSimpleFilterOption,
    ETrainsFilterType.STATION_FROM
> {
    type: ETrainsFilterType.STATION_FROM = ETrainsFilterType.STATION_FROM;

    apply(value: string[], {stationFrom}: ITrainsTariffApiSegment): boolean {
        return value.length
            ? value.includes(stationFrom?.id?.toString())
            : true;
    }

    updateOptions(
        options: ITrainsSimpleFilterOption[],
        {stationFrom}: ITrainsTariffApiSegment,
    ): ITrainsSimpleFilterOption[] {
        const fromId =
            stationFrom && stationFrom.id && stationFrom.id.toString();

        if (fromId && !options.find(item => item.value === fromId)) {
            options.push({
                value: fromId,
                text: stationFrom.popularTitle || stationFrom.title,
            });
        }

        return options;
    }

    /**
     * Возвращает список опций из данных обо всех фильтрах (из state).
     */
    getOptionsFromFiltersData(
        filtersData: ITrainsFilters,
    ): ITrainsSimpleFilterOption[] {
        return filtersData[this.type].options;
    }

    /**
     * Возвращает начальное состояние данного фильтра.
     */
    initFilterData(
        segments: ITrainsTariffApiSegment[],
    ): ITrainsStationFromFilter {
        const options = this.getOptions(segments);
        const availableWithOptions = this.isAvailableWithOptions(options);
        const filteredSegmentIndices: boolean[] = new Array(
            segments.length,
        ).fill(true);

        return {
            value: this.getDefaultValue(),
            options,
            activeOptions: options,
            availableWithOptions,
            availableWithActiveOptions: availableWithOptions,
            type: this.type,
            filteredSegmentIndices,
        };
    }

    deserializeFromQuery({stationFrom}: ITrainsSearchLocation): string[] {
        if (!stationFrom) {
            return this.getDefaultValue();
        }

        return Array.isArray(stationFrom) ? stationFrom : [stationFrom];
    }

    serializeToQuery(value: string[]): {stationFrom: string[]} {
        return {stationFrom: value};
    }

    validateValue(
        value: string[],
        options: ITrainsSimpleFilterOption[],
    ): string[] {
        const validValues = new Set<string>(options.map(item => item.value));

        return value.filter(item => validValues.has(item));
    }

    formatOptions(
        options: ITrainsSimpleFilterOption[],
        formatTextFn?: (
            option: ITrainsSimpleFilterOption,
            index: number,
        ) => string,
    ): ITrainsSimpleFilterOption[] {
        return formatTextFn
            ? this.formatTextWithFn(options, formatTextFn)
            : options;
    }
}

export default new StationFrom();
