import {
    ETrainsFilterType,
    ITrainsFilters,
    ITrainsSimpleFilterOption,
    ITrainsStationToFilter,
} from 'types/trains/search/filters/ITrainsFilters';
import {ITrainsSearchLocation} from 'types/trains/search/ITrainsSearchLocation';
import {ITrainsTariffApiSegment} from 'server/api/TrainsApi/types/ITrainsGetTariffsApi/models';

import BaseListManager from 'projects/trains/lib/filters/managers/utilities/BaseListManager';

export class StationTo extends BaseListManager<
    string,
    ITrainsSimpleFilterOption,
    ETrainsFilterType.STATION_TO
> {
    type: ETrainsFilterType.STATION_TO = ETrainsFilterType.STATION_TO;

    apply(value: string[], {stationTo}: ITrainsTariffApiSegment): boolean {
        return value.length ? value.includes(stationTo?.id?.toString()) : true;
    }

    updateOptions(
        options: ITrainsSimpleFilterOption[],
        {stationTo}: ITrainsTariffApiSegment,
    ): ITrainsSimpleFilterOption[] {
        const toId = stationTo && stationTo.id && stationTo.id.toString();

        if (toId && !options.find(item => item.value === toId)) {
            options.push({
                value: toId,
                text: stationTo.popularTitle || stationTo.title,
            });
        }

        return options;
    }

    /**
     * Возвращает список опций из данных обо всех фильтрах (из state).
     */
    getOptionsFromFiltersData(
        filtersData: ITrainsFilters,
    ): ITrainsSimpleFilterOption[] {
        return filtersData[this.type].options;
    }

    /**
     * Возвращает начальное состояние данного фильтра.
     */
    initFilterData(
        segments: ITrainsTariffApiSegment[],
    ): ITrainsStationToFilter {
        const options = this.getOptions(segments);
        const availableWithOptions = this.isAvailableWithOptions(options);
        const filteredSegmentIndices: boolean[] = new Array(
            segments.length,
        ).fill(true);

        return {
            value: this.getDefaultValue(),
            options,
            activeOptions: options,
            availableWithOptions,
            availableWithActiveOptions: availableWithOptions,
            type: this.type,
            filteredSegmentIndices,
        };
    }

    deserializeFromQuery({stationTo}: ITrainsSearchLocation): string[] {
        if (!stationTo) {
            return this.getDefaultValue();
        }

        return Array.isArray(stationTo) ? stationTo : [stationTo];
    }

    serializeToQuery(value: string[]): {stationTo: string[]} {
        return {stationTo: value};
    }

    validateValue(
        value: string[],
        options: ITrainsSimpleFilterOption[],
    ): string[] {
        const validValues = new Set<string>(options.map(item => item.value));

        return value.filter(item => validValues.has(item));
    }

    formatOptions(
        options: ITrainsSimpleFilterOption[],
        formatTextFn?: (
            option: ITrainsSimpleFilterOption,
            index: number,
        ) => string,
    ): ITrainsSimpleFilterOption[] {
        return formatTextFn
            ? this.formatTextWithFn(options, formatTextFn)
            : options;
    }
}

export default new StationTo();
